//
//  AudioBufferPool.hpp
//  SwitchboardSDK
//
//  Created by Balázs Kiss on 2022. 03. 26..
//  Copyright © 2022. Synervoz Inc. All rights reserved.
//

#pragma once

#include "AudioBuffer.hpp"
#include "Switchboard.hpp"

#include <optional>
#include <vector>

namespace switchboard {

/**
 * AudioBufferPool class.
 * @brief Allocates memory for multiple AudioBuffer instances.
 * @details This class can be used to avoid allocations on the real-time audio thread by pre-allocating enough memory for the audio buffers. Only supports non-interleaved buffers.
 */
template <typename T>
class AudioBufferPool {
public:
    /**
     * @brief Creates an AudioBufferPool instance.
     *
     * @param maxNumberOfBuffers Maximum number of AudioBuffer instances that can be taked from the pool.
     * @param maxNumberOfChannels Maximum number of channels for each audio buffer.
     * @param maxNumberOfFrames Maximum number of channels for each audio buffer.
     */
    AudioBufferPool(const uint maxNumberOfBuffers, const uint maxNumberOfChannels, const uint maxNumberOfFrames);

    /**
     * @brief AudioBufferPool destructor.
     */
    ~AudioBufferPool();

    /**
     * @brief Returns the number of available audio buffers in the pool.
     *
     * @returns The number of available audio buffers.
     */
    uint getNumberOfAvailableBuffers() const;

    /**
     * @brief Takes an audio buffer from the pool.
     *
     * @param numberOfChannels The number of channels for the audio buffer.
     * @param numberOfFrames The number of frames for the audio buffer.
     *
     * @returns The AudioBuffer instance.
     */
    AudioBuffer<T>& takeAudioBuffer(const uint numberOfChannels, const uint numberOfFrames);

    /**
     * @brief Returns an audio buffer to the pool.
     *
     * @param audioBuffer The returned audio buffer.
     */
    void returnAudioBuffer(AudioBuffer<T>& audioBuffer);

private:
    uint maxNumberOfBuffers;
    uint maxNumberOfChannels;
    std::vector<T**> dataBuffers;
    std::vector<std::optional<AudioBuffer<T>>> audioBuffers;
};

}
