//
// Created by Nádor Iván on 2024. 05. 27..
// Copyright © 2024. Synervoz Inc. All rights reserved.
//

#pragma once

#include "AudioData.hpp"
#include "AudioGraph.hpp"
#include "AudioIO.hpp"
#include "Switchboard.hpp"

#include <memory>

/**
 * AudioEngine class for desktop platforms
 * @brief Provides realtime audio processing through AudioGraph on macOS, Linux and Windows.
 */
class AudioEngine {
public:
    /**
     * @brief AudioEngine constructor.
     *
     * @param api The requested audio API of the system.
     */
    AudioEngine(switchboard::AudioIO::AudioAPI api);

    /**
     * @brief AudioEngine destructor.
     */
    ~AudioEngine();

    /**
     * @brief Get the available audio devices on the system.
     *
     * @returns The available audio devices.
     */
    const std::vector<switchboard::AudioIO::AudioDevice> getAudioDevices();

    /**
     * @brief Get the current input audio device.
     *
     * @returns The current input audio device.
     */
    const std::optional<switchboard::AudioIO::AudioDevice> getCurrentInputDevice();

    /**
     * @brief Get the current output audio device.
     *
     * @returns The current output audio device.
     */
    const std::optional<switchboard::AudioIO::AudioDevice> getCurrentOutputDevice();

    /**
     * @brief Starts the audio engine.
     *
     * @param audioGraph The audio graph that should process the audio data in `process`.
     * @param streamParameters The requested stream parameters of the audio stream.
     *
     * @returns AudioIONoError if started successfully or an AudioIO error type if not.
     */
    switchboard::AudioIO::Error start(
        switchboard::AudioGraph* audioGraph,
        switchboard::AudioIO::StreamParameters streamParameters = switchboard::AudioIO::StreamParameters()
    );

    /**
     * @brief Stops the audio engine.
     */
    void stop();

    /**
     * @brief Gets whether the audio engine is running currently.
     *
     * @returns True if the audio engine is running.
     */
    bool getIsRunning() const;

    /**
     * @brief Processes a chunk of audio data through the audio graph.
     *
     * @param inputBuffers The input audio data in stacked float buffers.
     * @param numberOfInputChannels The number of input channels.
     * @param outputBuffers The output audio data in stacked float buffers.
     * @param numberOfOutputChannels The number of output channels.
     * @param numberOfFrames The number of frames (single channel) in the buffers.
     * @param sampleRate The sample rate of the audio data
     *
     * @returns True if the processing was successful.
     */
    bool process(
        float** inputBuffers,
        uint numberOfInputChannels,
        float** outputBuffers,
        uint numberOfOutputChannels,
        uint numberOfFrames,
        uint sampleRate
    );

private:
    std::atomic<bool> isRunning;
    switchboard::AudioGraph* audioGraph;
    std::unique_ptr<switchboard::AudioIO> audioIO;

    static void audioProcessingCallback(float**, uint, float**, uint, uint, uint, void*);
};
