//
//  AudioGraph.hpp
//  SwitchboardSDK
//
//  Created by Balázs Kiss on 2022. 02. 24..
//  Copyright © 2022. Synervoz Inc. All rights reserved.
//

#pragma once

#include "AudioBuffer.hpp"
#include "AudioBufferPool.hpp"
#include "AudioGraphInputNode.hpp"
#include "AudioGraphNodeState.hpp"
#include "AudioGraphOutputNode.hpp"
#include "AudioProcessorNode.hpp"
#include "Graph.hpp"
#include "Switchboard.hpp"
#include "SwitchboardObject.hpp"

#include <map>
#include <memory>
#include <set>

namespace switchboard {

/**
 * AudioGraph class.
 * @brief Manages a graph of AudioNode instances.
 * @details This class can be used to easily put together complicated audio pipelines by adding and then connecting AudioNode instances together.
 */
class AudioGraph : public SwitchboardObject {
public:
    WASM_EXPORT(AudioGraph);

    /**
     * @brief Maximum number of channels that is supported by the audio graph.
     */
    const uint maxNumberOfChannels;

    /**
     * @brief Maximum number of frames that is supported by the audio graph.
     */
    const uint maxNumberOfFrames;

    /**
     * @brief Creates an AudioGraph.
     *
     * @param maxNumberOfChannels Maximum number of channels that is supported by the audio graph.
     * @param maxNumberOfFrames Maximum number of frames that is supported by the audio graph.
     */
    WASM AudioGraph(
        const uint maxNumberOfChannels = constants::STEREO,
        const uint maxNumberOfFrames = constants::MAX_NUMBER_OF_FRAMES
    );

    /**
     * @brief AudioGraph destructor.
     */
    WASM ~AudioGraph();

    /**
     * @brief Returns the input node of the audio graph.
     *
     * @returns The input node.
     */
    WASM const AudioGraphInputNode& getInputNode() const;

    /**
     * @brief Returns the output node of the audio graph.
     *
     * @returns The output node.
     */
    WASM const AudioGraphOutputNode& getOutputNode() const;

    /**
     * @brief Adds a node to the audio graph.
     *
     * @param audioNode The node to be added.
     * @param moveOwnership If set to true, the AudioGraph will destruct the audio node.
     *
     * @returns True if the node was successfully added, false otherwise.
     */
    WASM bool addNode(AudioNode& audioNode, bool moveOwnership = false);

    /**
     * @brief Removes a node from the audio graph.
     *
     * @param audioNode The node to be removed.
     *
     * @returns True if the node was successfully removed, false otherwise.
     */
    WASM bool removeNode(AudioNode& audioNode);

    /**
     * @brief Gets a list of all audio nodes in the audio graph.
     *
     * @returns A vector with all the audio nodes in the audio graph.
     */
    const std::vector<AudioNode*>& getNodes() const;

    /**
     * @brief Finds an audio node by name.
     *
     * @param name The name of the audio node.
     *
     * @returns The found audio node, or nullptr if the audio node could not be found.
     */
    AudioNode* getNodeByName(const std::string& name) const;

    /**
     * @brief Finds an audio node by name.
     *
     * @param name The name of the audio node.
     *
     * @returns The found audio node, or nullptr if the audio node could not be found.
     */
    WASM AudioNode* getNodeByName(const char* name) const;

    /**
     * @brief Connects a source node to a processor node.
     *
     * @param srcNode The source node.
     * @param dstNode The processor node.
     *
     * @returns True if the nodes were successfully connected, false otherwise.
     */
    WASM bool connect(const AudioSourceNode& srcNode, const AudioProcessorNode& dstNode);

    /**
     * @brief Connects a source node to a sink node.
     *
     * @param srcNode The source node.
     * @param dstNode The sink node.
     *
     * @returns True if the nodes were successfully connected, false otherwise.
     */
    WASM bool connect(const AudioSourceNode& srcNode, const AudioSinkNode& dstNode);

    /**
     * @brief Connects a processor node to another processor node.
     *
     * @param srcNode The source processor node.
     * @param dstNode The destination processor node.
     *
     * @returns True if the nodes were successfully connected, false otherwise.
     */
    WASM bool connect(const AudioProcessorNode& srcNode, const AudioProcessorNode& dstNode);

    /**
     * @brief Connects a processor node to a sink node.
     *
     * @param srcNode The processor node.
     * @param dstNode The sink node.
     *
     * @returns True if the nodes were successfully connected, false otherwise.
     */
    WASM bool connect(const AudioProcessorNode& srcNode, const AudioSinkNode& dstNode);

    /**
     * @brief Starts the audio graph.
     *
     * @returns True if the audio graph was started successfully, false otherwise.
     */
    WASM bool start();

    /**
     * @brief Stops the audio graph
     */
    WASM void stop();

    /**
     * @brief Checks if the audio graph has been started.
     *
     * @returns True if the audio graph was already started successfully, false otherwise.
     */
    WASM bool isRunning();

    /**
     * @brief The real-time audio process function of the audio graph.
     * @details Processes the audio in the provided input audio buses and puts the output audio in the provided output audio buses.
     *
     * @param inAudioBuses The input audio buses.
     * @param outAudioBuses The output audio buses.
     *
     * @returns True if the process call was successful, false otherwise.
     */
    WASM bool process(AudioBusList& inAudioBuses, AudioBusList& outAudioBuses) NONBLOCKING;

    /**
     * @brief The real-time audio process function of the audio graph that processes audio buffers.
     * @details Processes the audio in the provided input audio buffer and puts the output audio in the provided output audio buffer. This is a convenience method that can be used when the audio graph has no more than one input and output buses. To process multiple buses, use the `process` method.
     *
     * @param inAudioBuffer The input audio buffer, or null if there graph has no input.
     * @param outAudioBuffer The output audio buffer, or null if the graph has no output.
     *
     * @returns True if the process call was successful, false otherwise.
     */
    WASM bool processBuffer(AudioBuffer<float>* inAudioBuffer, AudioBuffer<float>* outAudioBuffer);

    // MARK: Overridden methods

    Result<void> setValue(const std::string& key, const std::any& value) override;
    Result<std::any> getValue(const std::string& key) override;
    Result<std::any> callAction(const std::string& actionName, const std::map<std::string, std::any>& params) override;

private:
    AudioGraphInputNode inputNode;
    AudioGraphOutputNode outputNode;
    Graph graph;

    std::vector<AudioNode*> nodesArray;
    std::vector<std::unique_ptr<AudioNode>> ownedNodes;
    std::map<AudioNode*, uint> nodes;
    std::vector<AudioGraphNodeConnection> connections;
    std::unique_ptr<AudioBufferPool<float>> bufferPool;
    bool isStarted;
    std::vector<uint> nodeExecutionOrder;
    std::map<uint, AudioGraphNodeState*> nodeStates;
    std::unique_ptr<AudioBusFormatList> previousInputFormat;
    std::unique_ptr<AudioBusFormatList> previousOutputFormat;
    std::vector<uint> processQueue;

    bool hasNode(const AudioNode& node) const;
    bool connectInternal(const AudioNode& srcNode, const AudioNode& dstNode);
    AudioNode* getNode(const uint nodeID) const;
    void setNumberOfAudioBusesForAllNodes();
    void createNodeExecutionOrder();
    bool setAudioFormatForNode(AudioNode& node);
    bool setAudioFormatForAllNodes(AudioBusList& inAudioBuses, AudioBusList& outAudioBuses);
    void prepareBuffers(const AudioBusList& inAudioBuses, const AudioBusList& outAudioBuses);
    void resetNodeStates();
    void returnAllAudioBuffers();
    void executeNode(const uint nodeID);
    void executeNodes();

    AudioGraph(const AudioGraph&) = delete;
    AudioGraph& operator=(const AudioGraph&) = delete;
};

}
