//
//  AudioGraphFactory.hpp
//  SwitchboardSDK
//
//  Created by Balazs Kiss on 2023. 07. 04..
//

#pragma once

#include "AudioGraph.hpp"
#include "AudioNodeFactory.hpp"

#include <memory>
#include <vector>

namespace switchboard {

/**
 * AudioGraphFactory class.
 * @brief Creates AudioGraph instances.
 */
class AudioGraphFactory {
public:
    WASM_EXPORT(AudioGraphFactory);

    /**
     * @brief Registers a node factory that can create audio nodes for the audio graph.
     *
     * @param nodeFactory The added AudioNodeFactory.
     */
    static void addNodeFactory(const std::shared_ptr<AudioNodeFactory> nodeFactory);

    /**
     * @brief Parses a JSON audio graph config and returns the creates AudioGraph.
     * @details The returned audio graph instances need to be freed by the user of the AudioGraphFactory.
     *
     * @param config The JSON audio graph config.
     *
     * @returns The created AudioGraph instance, or nullptr if the audio graph could not be created.
     */
    static AudioGraph* parseJSON(const std::string& config);

    /**
     * @brief Parses a JSON audio graph config and returns the creates AudioGraph.
     * @details The returned audio graph instances need to be freed by the user of the AudioGraphFactory.
     *
     * @param config The JSON audio graph config.
     *
     * @returns The created AudioGraph instance, or nullptr if the audio graph could not be created.
     */
    WASM static AudioGraph* parseJSON(const char* config);

    /**
     * @brief Creates an audio graph from a map of parameters.
     * @details The returned audio graph instances need to be freed by the user of the AudioGraphFactory.
     * 
     * @param config The map of parameters.
     * 
     * @returns The created AudioGraph instance, or nullptr if the audio graph could not be created.
     */
    WASM static AudioGraph* createGraph(const std::map<std::string, std::any>& config);

private:
    static std::vector<std::shared_ptr<AudioNodeFactory>> nodeFactories;

    static bool connectNodes(AudioGraph* audioGraph, const AudioNode* srcNode, const AudioNode* dstNode);
    static AudioNode* createNode(const std::string& type, const std::map<std::string, std::any>& config);
};
}
