//
//  AudioNode.hpp
//  SwitchboardSDK
//
//  Created by Balázs Kiss on 2022. 03. 03..
//  Copyright © 2022. Synervoz Inc. All rights reserved.
//

#pragma once

#include "Parameter.hpp"
#include "Switchboard.hpp"
#include "SwitchboardObject.hpp"

#include <any>
#include <map>
#include <string>
#include <vector>

namespace switchboard {

/**
 * AudioNode class.
 * @brief Represents a node in the audio graph.
 */
class AudioNode : public SwitchboardObject {
public:
    /**
     * @brief AudioNode constructor.
     */
    AudioNode();

    /**
     * @brief AudioNode virtual destructor.
     */
    virtual ~AudioNode() = default;

    /// @brief The name of the audio node.
    std::string name;

    /**
     * @brief Gets the type of the audio node.
     *
     * @returns The type of the audio node.
     */
    const std::string& getType() const;

    /**
     * @brief Gets the display name of the audio node.
     * @details By default the display name follows this format: name (type)
     *
     * @returns The display name of the node.
     */
    virtual const std::string getDisplayName() const;

    /**
     * @brief Gets the parameters of the audio node.
     *
     * @returns A list of parameters that belong to the audio node.
     */
    virtual std::vector<std::unique_ptr<Parameter>>& getParameters();

    /**
     * @brief Gets the parameter with the given name.
     *
     * @param id The ID of the parameter.
     *
     * @returns Returns a pointer to the parameter, nullptr if not found.
     */
    virtual Parameter* getParameter(const std::string& id);

    // MARK: Overridden methods

    Result<void> setValue(const std::string& key, const std::any& value) override;
    Result<std::any> getValue(const std::string& key) override;
    Result<std::any> callAction(const std::string& actionName, const std::map<std::string, std::any>& params) override;

protected:
    std::string type;
    std::vector<std::unique_ptr<Parameter>> parameters;

    /**
     * @brief Initializes the node parameters from the given configuration values.
     *
     * @param config Configuration values for the parameters.
     */
    void initParameters(const std::map<std::string, std::any>& config);
};

}
