//
//  AudioProcessorNode.hpp
//  SwitchboardSDK
//
//  Created by Balázs Kiss on 2022. 06. 03..
//  Copyright © 2022. Synervoz Inc. All rights reserved.
//

#pragma once

#include "AudioBusFormatList.hpp"
#include "AudioBusList.hpp"
#include "AudioNode.hpp"

namespace switchboard {

/**
 * AudioProcessorNode class.
 * @brief A node that processes audio and has both incoming and outgoing connections.
 */
class AudioProcessorNode : public AudioNode {
public:
    /**
     * @brief AudioProcessorNode virtual destructor.
     */
    virtual ~AudioProcessorNode() = default;

    /**
     * @brief Sets number of buses.
     * @details Should be overridden by subclasses
     *
     * @param numberOfInputBuses The number of input buses to set.
     * @param numberOfOutputBuses The number of output buses to set.
     *
     * @returns True for valid number of buses, false otherwise.
     */
    virtual bool setNumberOfBuses(const uint numberOfInputBuses, const uint numberOfOutputBuses) = 0;

    /**
     * @brief Sets bus formats.
     * @details Should be overridden by subclasses. Please note that @ref setNumberOfBuses is guaranteed to be called before this method.
     *
     *
     * @param inputBusFormats The input bus formats to set.
     * @param outputBusFormats The output bus formats to set.
     *
     * @returns True if the bus formats were set successfully, false otherwise.
     */
    virtual bool setBusFormats(AudioBusFormatList& inputBusFormats, AudioBusFormatList& outputBusFormats) = 0;

    /**
     * @brief Processes audio data from the given input buses and produces the audio output to the given output buses.
     * @details Should be overridden by subclasses.
     *
     * @param inBuses The input buses that contain the audio data.
     * @param outBuses The output buses which should be filled with the processed audio data.
     *
     * @returns True if audio was processed successfully, false otherwise.
     */
    virtual bool process(AudioBusList& inBuses, AudioBusList& outBuses) = 0;
};

}
