//
//  FloatToShortInt.hpp
//  SwitchboardSDK
//
//  Created by Balazs Kiss on 2023. 04. 04..
//

#pragma once

#include "AudioBuffer.hpp"
#include "Limit.hpp"
#include "Switchboard.hpp"

#include <climits>

namespace switchboard {

/**
 * @brief Converts a float (32 bit) audio value to short int (16 bit).
 * @details The conversion happens in the sense of audio values, meaning float has a range of -1.0...1.0 and short int has -32768...32767.
 *
 * @param value The input value in float.
 *
 * @returns The value in short.
 */
inline int16 floatToShortInt(float value) {
    return static_cast<int16>(limit<float>(-1.0, 1.0, value) * SHRT_MAX);
}

/**
 *  @brief Converts float buffer to int16 buffer.
 *  @details Input and output buffers must be the same length.
 *
 *  @param inputBuffer The float buffer that needs to be converted.
 *  @param outputBuffer The int16 buffer that will store the converted data.
 *  @param numberOfSamples Length of input and output buffers.
 */
void floatToShortInt(const float* inputBuffer, int16* outputBuffer, const uint numberOfSamples);

/**
 *  @brief Converts float AudioBuffer to int16 AudioBuffer.
 *  @details Input and output buffers must have the same number of channels and same number of frames.
 *
 *  @param inputBuffer The float AudioBuffer that needs to be converted.
 *  @param outputBuffer The int16 AudioBuffer that will store the converted data.
 */
void floatToShortInt(const AudioBuffer<float>& inputBuffer, AudioBuffer<int16>& outputBuffer);

}
