//
//  Generator.hpp
//  SwitchboardSDK
//
//  Created by Pesti József on 19/10/2023.
//

#pragma once

#include "AudioBuffer.hpp"
#include "GeneratorType.hpp"
#include "Ramp.hpp"
#include "Switchboard.hpp"

#include <atomic>

namespace switchboard {

/**
 * Generator base class
 * @brief Provides general interface for signal generators
 */
class Generator {
public:
    /**
     * @brief Generator constructor.
     *
     * @param frequency The initial frequency in Hz. Default value is 440 Hz.
     */
    Generator(const float frequency = 440.0f);

    /**
     * @brief Generator copy constructor.
     */
    Generator(const Generator& generator);

    /**
     * @brief Generator destructor.
     */
    virtual ~Generator() = default;

    /**
     * @brief Gets the current frequency value of the generated signal.
     *
     * @returns The frequency value in Hz.
     */
    float getFrequency() const;

    /**
     * @brief Sets the frequency of the generated signal.
     *
     * @param frequency The new frequency value in Hz.
     */
    void setFrequency(const float frequency);

    /**
     * @brief Gets the current amplitude value of the generated signal.
     *
     * @returns The amplitude value.
     */
    float getAmplitude() const;

    /**
     * @brief Sets the amplitude of the generated signal.
     *
     * @param amplitude The new amplitude value. Should be between 0 and 1.
     */
    void setAmplitude(const float amplitude);

    /**
     * @brief Gets the type of the Generator.
     *
     * @returns The type of the Generator.
     */
    GeneratorType getType();

    /**
     * @brief Generates the audio data into a data buffer.
     *
     * @param buffer The data buffer.
     * @param numberOfSamples The number of samples to generate.
     * @param sampleRate The sample rate.
     */
    template <typename T>
    void generate(T* buffer, const uint numberOfSamples, const uint sampleRate);

    /**
     * @brief Generates the audio data into an audio buffer.
     *
     * @param audioBuffer The audio buffer.
     */
    template <typename T>
    void generate(AudioBuffer<T>& audioBuffer);

protected:
    float phase;
    GeneratorType type;
    void stepPhase();

private:
    /*
     * @brief Should generate a number between -1.0f and 1.0f
     */
    virtual float generateSample() = 0;
    std::atomic<float> amplitude;
    std::atomic<float> frequency;
    Ramp amplitudeRamp;
};
}
