//
//  MusicDuckingNode.hpp
//  SwitchboardAudio
//
//  Created by Balazs Kiss on 2022. 12. 13..
//  Copyright © 2022. Synervoz Inc. All rights reserved.
//

#pragma once

#include "AudioData.hpp"
#include "AudioProcessorNode.hpp"
#include "DuckingManager.hpp"
#include "SimpleDuckingCompressor.hpp"
#include "VoiceActivityDetector.hpp"

#include <memory>

namespace switchboard {

/**
 * MusicDuckingNode class.
 * @brief Ducks a stereo music signal based on mono voice signal(s). Music signal has to be connected to the first input bus, ducking voice signals have to be connected to the second, third, ... input buses. Output is the ducked music signal.`
 */
class MusicDuckingNode : public AudioProcessorNode {
public:
    WASM_EXPORT(MusicDuckingNode)

    /**
     * @brief MusicDuckingNode constructor.
     */
    WASM MusicDuckingNode();

    /**
     * @brief MusicDuckingNode destructor.
     */
    WASM ~MusicDuckingNode();

    /**
     * @brief Gets the current ducking amount.
     * Returned value is between 0 and 1.
     *
     * @param duckingSignalIndex Index of the ducking signal.

     * @return The current ducking amount.
     */
    WASM float getDuckingAmount(const uint duckingSignalIndex) const;

    /**
     *  @brief Sets the current ducking amount.
     *  The provided value has to be between 0 and 1.
     *
     *  @param newDuckingAmount The new amount.
     *  @param duckingSignalIndex Index of the trigger input.
     */
    WASM void setDuckingAmount(const float newDuckingAmount, const uint duckingSignalIndex);

    /**
     * @brief Gets the ducking release amount.
     * When the ducking threshold is not exceeded, the ducking amount value will be increased by this value.
     * Default value: 0.5 dB
     *
     * @returns the ducking release amount.
     */
    WASM float getDuckReleaseAmount() const;

    /**
     * @brief Sets the ducking release amount.
     * When the ducking threshold is not exceeded, the ducking amount value will be increased by this value.
     * Default value: 0.5 dB
     *
     * @param newDuckReleaseAmount The new value.
     */
    WASM void setDuckReleaseAmount(const float newDuckReleaseAmount);

    /**
     * @brief Gets the number of seconds to hold ducking value.
     * When ducking stops being triggered, ducking will still be held for this many seconds.
     * Default value: 2.0 secs
     *
     * @returns The number of seconds to hold ducking.
     */
    WASM float getNumSecondsToHoldDucking() const;

    /**
     * @brief Sets the number of seconds to hold ducking value.
     *
     * @param numSecondsToHoldDucking The new value.
     */
    WASM void setNumSecondsToHoldDucking(const float numSecondsToHoldDucking);

    /**
     * @brief Tells if the node is in mixDuckingSignal mode.
     * If set to yes, audio trigger input will be mixed to the output.
     *
     * @returns True if the node is in passthrough mode.
     */
    WASM bool getMixDuckingSignal() const;

    /**
     * @brief Sets the node to mix ducking signal mode..
     *
     * @param mix The new value.
     */
    WASM void setMixDuckingSignal(const bool mix);

    /**
     * @brief Gets the pointer to the compressor that does the ducking.
     *
     * @returns A pointer to the compressor instance.
     */
    DuckingCompressor* getCompressor() const;

    /**
     * @brief Sets the compressor that is used by the ducking node.
     *
     * @param compressor Pointer to a DuckingCompressor instance.
     */
    void setCompressor(DuckingCompressor* compressor);

    // MARK: Overridden methods

    bool setNumberOfBuses(const uint numberOfInputBuses, const uint numberOfOutputBuses) override;
    bool setBusFormats(AudioBusFormatList& inputBusFormats, AudioBusFormatList& outputBusFormats) override;
    bool process(AudioBusList& inBuses, AudioBusList& outBuses) override;

private:
    std::vector<std::unique_ptr<VoiceActivityDetector>> vad;
    std::unique_ptr<DuckingManager> duckingManager;
    AudioData<float> tempInterleavedBuffer;
    SimpleDuckingCompressor duckingCompressor;
    std::atomic<bool> mixDuckingSignal;

    void createParameters();
    float getFirstDuckingAmount() const;
    void setDuckingAmounts(const float newDuckingAmount);
};

}
