//
//  RecorderNode.hpp
//  SwitchboardSDK
//
//  Created by Nádor Iván on 2022. 09. 01..
//

#pragma once

#include "AudioData.hpp"
#include "AudioRingBuffer.hpp"
#include "Codec.hpp"
#include "Counter.hpp"
#include "FileWriter.hpp"
#include "Resampler.hpp"
#include "SingleBusAudioSinkNode.hpp"

#include <atomic>
#include <condition_variable>
#include <mutex>
#include <thread>

namespace switchboard {

/**
 * RecorderNode class
 * @brief Records the incoming audio data to a file.
 */
class RecorderNode : public SingleBusAudioSinkNode {
public:
    /**
     * @brief Creates a RecorderNode instance.
     *
     * @param recordingSampleRate The sample rate which the output file will be saved with. SAMPLE_RATE_UNDEFINED (0) means, sample rate will be set to the first received bus sample rate in setBusFormat.
     * @param numberOfRecordedChannels Number of channels to record. Must match the number of channels of the incoming audio buffer.
     */
    RecorderNode(
        const uint recordingSampleRate = constants::SAMPLE_RATE_UNDEFINED,
        const uint numberOfRecordedChannels = constants::STEREO
    );

    /**
     * @brief RecorderNode destructor.
     */
    ~RecorderNode();

    /**
     * @brief Start recording the incoming audio.
     */
    void start();

    /**
     * @brief Stop recording the incoming audio and write it to file.
     *
     * @param recordingFilePath Final recording file path.
     * @param format Final recording file format.
     */
    bool stop(const std::string& recordingFilePath, Codec format);

    /**
     * @brief Check whether the RecorderNode is recording.
     *
     * @returns True if the RecorderNode is recording.
     */
    bool getIsRecording() const;

    /**
     * @brief Set whether the RecorderNode should run in realtime mode.
     *
     * @param isRealtime True if the RecorderNode should run in realtime mode.
     */
    void setIsRealtime(const bool isRealtime);

    /**
     * @brief Check whether the RecorderNode is running in realtime mode.
     *
     * @returns True if the RecorderNode is running in realtime mode.
     */
    bool getIsRealtime() const;

    // MARK: Overridden methods

    bool setBusFormat(AudioBusFormat& busFormat) override;
    bool consume(AudioBus& bus) override;

private:
    static Counter counter;

    std::string name;

    uint numberOfRecordedChannels;
    std::unique_ptr<AudioRingBuffer<float>> recorderRingBuffer;
    AudioData<float> temporaryRecorderData;
    AudioBuffer<float> temporaryRecorderBuffer;
    AudioData<float> temporaryInterleavedData;
    AudioBuffer<float> temporaryInterleavedBuffer;

    std::unique_ptr<FileWriter> temporaryFileWriter;
    std::string temporaryFilePath;

    std::atomic<uint> currentInputSampleRate;
    uint recordingSampleRate;
    std::unique_ptr<Resampler> resampler;
    AudioData<float> temporaryResampledData;
    AudioBuffer<float> temporaryResampledBuffer;

    std::thread recorderThread;
    std::condition_variable recorderThreadConditionVariable;
    std::mutex recorderThreadMutex;
    std::atomic<bool> stopRecorderThread;

    std::atomic<bool> isRecording;

    // Flag indicates whether the Recorder is delegating file writing to a worker thread
    bool isRealtime;

    void recorderThreadProcess();
    void stopThread();

    void appendDataToTemporaryFile();

    bool saveFile(const std::string& recordingFilePath, Codec format);
};

}
