//  Copyright © 2019. Synervoz Inc. All rights reserved.

#pragma once

#include "SBJNIObjectMapping.hpp"

#include <jni.h>
#include <memory>
#include <string>

/**
 * Switchboard Java Native Interface
 * Provides JNI helper methods for building the bridge between C++ and Java.
 */
class SBJNI {

public:
    /**
     * Stores a Java and C++ shared pointer mapping in a Java class.
     *
     * @tparam T Type of the C++ object.
     * @param env The JNI environment.
     * @param jObject The Java object.
     * @param ptr The C++ shared pointer.
     */
    template <typename T>
    inline static void storeNativeSharedPtrInJavaObject(JNIEnv* env, jobject jObject, std::shared_ptr<T> ptr) {
        jobject globalJObject = env->NewGlobalRef(jObject);
        auto holder = new SBJNIObjectMapping<T>(globalJObject, ptr);
        setHandle(env, jObject, holder);
    }

    /**
     * Stores a Java and C++ pointer mapping in a Java class.
     *
     * @tparam T Type of the C++ object.
     * @param env The JNI environment.
     * @param jObject The Java object.
     * @param ptr The C++ pointer.
     */
    template <typename T>
    inline static void storeNativePtrInJavaObject(JNIEnv* env, jobject jObject, T* ptr) {
        jobject globalJObject = env->NewGlobalRef(jObject);
        auto holder = new SBJNIObjectMapping<T>(globalJObject, ptr);
        setHandle(env, jObject, holder);
    }

    /**
     * Disposes the mapping object inside a Java object.
     *
     * @tparam T The type of the C++ object.
     * @param env The JNI environment.
     * @param jObject The Java object.
     */
    template <typename T>
    inline static void disposeNative(JNIEnv* env, jobject jObject) {
        auto holder = getHandle<SBJNIObjectMapping<T>>(env, jObject);
        jobject globalJObject = holder->getJavaObject();
        env->DeleteGlobalRef(globalJObject);

        delete holder;

        setHandle(env, jObject, static_cast<T*>(nullptr));
    }

    /**
     * Gets the mapped C++ object from the Java object.
     *
     * @tparam T The type of the C++ object.
     * @param env The JNI environment.
     * @param jObject The Java object.
     *
     * @returns Pointer to the C++ object.
     */
    template <typename T>
    inline static T* getCppObject(JNIEnv* env, jobject jObject) {
        SBJNIObjectMapping<T>* holder = getHandle<SBJNIObjectMapping<T>>(env, jObject);
        return holder->getCppObject();
    }

    /**
     * Creates a C++ string from a Java string.
     *
     * @param env The JNI environment.
     * @param javaString The Java string to convert.
     *
     * @returns The C++ string.
     */
    static std::string createCppStringFromJavaString(JNIEnv* env, jstring javaString) {
        std::string result;
        if (javaString == nullptr) {
            return result;
        }
        const char* utf8Chars = env->GetStringUTFChars(javaString, NULL);
        result = utf8Chars;
        env->ReleaseStringUTFChars(javaString, utf8Chars);
        return result;
    }

    /**
     * Creates a Java string from a C++ string.
     *
     * @param env The JNI environment.
     * @param javaString The C++ string to convert.
     *
     * @returns The Java string.
     */
    static jstring createJavaStringFromCppString(JNIEnv* env, const char* cppString) {
        return env->NewStringUTF(cppString);
    }

    /**
     * Gets the C++ object from the pointer memory location.
     *
     * @tparam T The type of the C++ object.
     * @param pointer The memory address of the C++ object.
     *
     * @returns Pointer to the C++ object.
     */
    template <typename T>
    static T* getCppObjectFromPointerAddress(jlong pointer) {
        auto* jniHolder = reinterpret_cast<SBJNIObjectMapping<T>*>(pointer);
        return jniHolder->getCppObject();
    }

    /**
     * @brief Gets the JNIEnv of the current thread
     *
     * @details The JNI interface pointer (JNIEnv) is valid only in the current thread, so we should
     * use the JavaVM pointer instead to retrieve the JNIEnv associated with the current thread.
     * @tparam T The type of the C++ object.
     * @param pointer The memory address of the C++ object.
     *
     * @returns Pointer to the JNIEnv object.
     */
    static JNIEnv* getThreadLocalEnv(JavaVM* jvm) {
        JNIEnv* environment;
        int getEnvStat = jvm->GetEnv((void**)&environment, JNI_VERSION_1_6);
        if (getEnvStat == JNI_EDETACHED) {
#ifdef __ANDROID__
            jvm->AttachCurrentThread(&environment, NULL);
#else
            jvm->AttachCurrentThread(reinterpret_cast<void**>(&environment), NULL);
#endif
        }
        return environment;
    }

private:
    inline static jfieldID getHandleField(JNIEnv* env, jobject jObject) {
        jclass jClass = env->GetObjectClass(jObject);
        const char* fieldName = "nativeHandle";
        // J is the type signature for long:
        return env->GetFieldID(jClass, fieldName, "J");
    }

    template <typename T>
    inline static T* getHandle(JNIEnv* env, jobject jObject) {
        jfieldID field = getHandleField(env, jObject);
        jlong handle = env->GetLongField(jObject, field);
        return reinterpret_cast<T*>(handle);
    }

    template <typename T>
    inline static void setHandle(JNIEnv* env, jobject jObject, T* holder) {
        jlong handle = reinterpret_cast<jlong>(holder);
        jfieldID field = getHandleField(env, jObject);
        env->SetLongField(jObject, field, handle);
    }
};