//
//  ShortIntToFloat.hpp
//  SwitchboardSDK
//
//  Created by Balazs Kiss on 2023. 04. 04..
//

#pragma once

#include "AudioBuffer.hpp"
#include "Switchboard.hpp"

#include <climits>

namespace switchboard {

/**
 * @brief Converts a short int (16 bit) audio value to float (32 bit).
 * @details The conversion happens in the sense of audio values, meaning float has a range of -1.0...1.0 and short int has -32768...32767.
 *
 * @param value The input value in short int.
 *
 * @returns The value in float.
 */
inline float shortIntToFloat(int16 value) {
    return static_cast<float>(value) / SHRT_MAX;
}

/**
 *  @brief Converts int16 buffer to float buffer.
 *  @details Input and output buffers must be the same length.
 *
 *  @param inputBuffer The int16 buffer that needs to be converted.
 *  @param outputBuffer The float buffer that will store the converted data.
 *  @param numberOfSamples Length of input and output buffers.
 */
void shortIntToFloat(const int16* inputBuffer, float* outputBuffer, const uint numberOfSamples);

/**
 *  @brief Converts int16 AudioBuffer to float AudioBuffer.
 *  @details Input and output buffers must have the same number of channels. Input and output buffers must have the same number of frames.
 *
 *  @param inputBuffer The int16 AudioBuffer that needs to be converted.
 *  @param outputBuffer The float AudioBuffer that will store the converted data.
 */
void shortIntToFloat(const AudioBuffer<int16>& inputBuffer, AudioBuffer<float>& outputBuffer);

}
