//
//  StringCallbackParameter.hp
//  SwitchboardSDK
//
//  Created by Iván Nádor on 2023. 07. 11..
//

#pragma once

#include "StringParameter.hpp"

namespace switchboard {

/**
 * @brief StringCallbackParameter class. A String parameter class that wraps a getter and a setter methods.
 */
template <typename T>
class StringCallbackParameter : public StringParameter {
public:
    typedef std::string (T::*StringGetter)() const;
    typedef void (T::*StringSetter)(std::string);

    /**
     * @brief StringCallbackParameter constructor.
     *
     * @param id The unique identifier of the parameter.
     * @param name The name of the parameter.
     * @param description The description of the parameter.
     * @param instance Pointer to the instance that the getter and setter methods belong to.
     * @param getter The getter method for the parameter.
     * @param setter The setter method for the parameter.
     */
    StringCallbackParameter(
        const std::string& id,
        const std::string& name,
        const std::string& description,
        T* instance,
        StringGetter getter,
        StringSetter setter
    ) :
        StringParameter(id, name, description),
        instance { instance },
        getter { getter },
        setter { setter } {}

    // MARK: Overridden methods

    std::string getValue() const override {
        return (instance->*getter)();
    }

    void setValue(const std::string newValue) override {
        (instance->*setter)(newValue);
    }

private:
    T* instance;
    StringGetter getter;
    StringSetter setter;
};

}
