//
//  Switchboard.hpp
//  SwitchboardSDK
//
//  Created by Balázs Kiss on 2022. 02. 22..
//  Copyright © 2022. Synervoz Inc. All rights reserved.
//

#pragma once

#include "SwitchboardVersion.hpp"

#include <algorithm>
#include <atomic>
#include <cassert>
#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <memory>
#include <string>

// MARK: - Types

// Check if data sizes are standard
static_assert(sizeof(char) == 1);
static_assert(sizeof(short) == 2);
static_assert(sizeof(int) == 4);
static_assert(sizeof(long long) == 8);

// Define fixed-size data types
using int8 = char;
using uint8 = unsigned char;
using int16 = short;
using uint16 = unsigned short;
using int64 = long long;
using uint64 = unsigned long long;
using uint = unsigned int;

// MARK: - Constants

namespace switchboard::constants {

static constexpr uint DEFAULT_SAMPLERATE = 44100;
static constexpr uint MAX_SAMPLERATE = 48000;
static constexpr uint MAX_BUFFER_DURATION_MS = 1000;
static constexpr uint MAX_NUMBER_OF_FRAMES = MAX_SAMPLERATE * MAX_BUFFER_DURATION_MS / 1000;
// ~85ms@48kHz or ~93ms@44.1kHz
static constexpr uint LEGACY_MAX_NUMBER_OF_FRAMES = 4096;

static constexpr uint NUM_CHANNELS_UNDEFINED = 0;
static constexpr uint MONO = 1;
static constexpr uint STEREO = 2;
static constexpr uint QUAD = 4;

static constexpr uint LEFT = 0;
static constexpr uint RIGHT = 1;

static constexpr uint SAMPLE_RATE_UNDEFINED = 0;
static constexpr uint SAMPLE_RATE_8KHz = 8000;
static constexpr uint SAMPLE_RATE_10KHz = 10000;
static constexpr uint SAMPLE_RATE_16KHz = 16000;
static constexpr uint SAMPLE_RATE_24KHz = 24000;
static constexpr uint SAMPLE_RATE_44_1KHz = 44100;
static constexpr uint SAMPLE_RATE_48KHz = 48000;

}

// MARK: - Preprocessor directives

// OS-specific macros
#if __APPLE__
#include <TargetConditionals.h>
#define SWITCHBOARD_APPLE 1

#if TARGET_OS_IOS
#define SWITCHBOARD_IOS 1
#define SWITCHBOARD_PLATFORM "ios"
#if TARGET_OS_SIMULATOR
// iOS, tvOS, or watchOS Simulator
#define SWITCHBOARD_IPHONE_SIMULATOR 1
#else
// iOS, tvOS, or watchOS device
#define SWITCHBOARD_IPHONE 1
#endif
#elif TARGET_OS_MACCATALYST
// Mac's Catalyst (ports iOS API into Mac, like UIKit).
#define SWITCHBOARD_MACCATALYST 1
#define SWITCHBOARD_PLATFORM "maccatalyst"
#elif TARGET_OS_OSX
#define SWITCHBOARD_MAC 1
#define SWITCHBOARD_PLATFORM "macos"
#elif TARGET_OS_WATCH
#define SWITCHBOARD_WATCHOS 1
#define SWITCHBOARD_IOS 1
#define SWITCHBOARD_PLATFORM "watchos"
#endif
#elif defined(__ANDROID__)
#define SWITCHBOARD_ANDROID 1
#define SWITCHBOARD_PLATFORM "android"
#elif defined(WIN32) || defined(_WIN32) || defined(__WIN32__) || defined(__NT__)
#define SWITCHBOARD_WINDOWS 1
#define SWITCHBOARD_PLATFORM "windows"
#elif defined(OS_LINUX) || defined(__linux__)
#define SWITCHBOARD_LINUX 1
#define SWITCHBOARD_PLATFORM "linux"
#elif defined(PLATFORM_WEB) || defined(__wasm__)
#define SWITCHBOARD_WEB 1
#define SWITCHBOARD_PLATFORM "web"
#elif defined(PLATFORM_EMBEDDED)
#define SWITCHBOARD_EMBEDDED 1
#define SWITCHBOARD_PLATFORM "embedded"
#else
#error "Unsupported platform"
#endif

// Apple-specific defines
#if defined(SWITCHBOARD_APPLE)
#if defined(__arm__) || defined(__arm64__)
#define SWITCHBOARD_ARM 1
#else
#define SWITCHBOARD_INTEL 1
#endif

#if defined(SWITCHBOARD_IOS)
#define USE_ACCELERATE
#define APPLE_DECODER_AVAILABLE
#endif
#endif

// Linux and Android-specific defines
#if defined(SWITCHBOARD_LINUX) || defined(SWITCHBOARD_ANDROID)
#if defined(__arm__) || defined(__arm64__) || defined(__aarch64__)
#define SWITCHBOARD_ARM 1
#elif __MMX__ || __SSE__ || __amd64__
#define SWITCHBOARD_INTEL 1
#endif

// Android-specific defines
#if defined(SWITCHBOARD_ANDROID)
#define USE_NEON
#endif
#endif

// Windows-specific defines
#if defined(SWITCHBOARD_WINDOWS)
#if defined(__arm__) || defined(__aarch64__) || defined(_M_ARM) || defined(_M_ARM64)
#define SWITCHBOARD_ARM 1
#else
#define SWITCHBOARD_INTEL 1
#endif
#endif

// Web-specific defines
#if defined(SWITCHBOARD_WEB)
#include <emscripten.h>
#define WASM EMSCRIPTEN_KEEPALIVE
#define WASM_EXPORT(CLASS_NAME) \
    WASM static size_t getObjectSize() { \
        return sizeof(CLASS_NAME); \
    }
#else
#define WASM
#define WASM_EXPORT(CLASS_NAME)
#endif

#if NETWORKING_ENABLED && SWITCHBOARD_LICENSE_VALIDATION_ENABLED
#define LICENSE_VALIDATION_ENABLED true
#else
#define LICENSE_VALIDATION_ENABLED false
#endif

#if !defined(M_PI)
#define M_PI 3.1415926535897932384626433832
#endif

#if !defined(M_PI_2)
#define M_PI_2 M_PI / 2
#endif

#if defined(__has_cpp_attribute) && __has_cpp_attribute(clang::nonblocking)
#define NONBLOCKING [[clang::nonblocking]]
#else
#define NONBLOCKING
#endif
