//
//  SwitchboardSDK.hpp
//  SwitchboardSDK
//
//  Created by Balazs Kiss on 2023. 01. 06..
//

#pragma once

#include "Switchboard.hpp"

#include <string>

namespace switchboard {

class LicenseManager;

/**
 * SwitchboardSDK class.
 * @brief Initializes and configures the Switchboard SDK.
 */
class SwitchboardSDK final {
public:
    WASM_EXPORT(SwitchboardSDK)

    WASM static void initialize(const char* appID, const char* appSecret);

    /**
     * Initializes the Switchboard SDK.
     *
     * @param appID The app ID value that identifies the user of the SDK.
     * @param appSecret The app secret ID.
     */
    static void initialize(const std::string appID, const std::string appSecret);

    /**
     * Gets whether the SDK has been initialized or not.
     *
     * @returns True if the SDK has been initialized, false otherwise.
     */
    static bool isInitialized();

    /**
     * Gets the version number of the SDK.
     *
     * @returns The version number of the SDK.
     */
    static std::string getVersionName();

    /**
     * Gets the build number of the SDK.
     *
     * @returns The build number of the SDK.
     */
    static uint getBuildNumber();

    /**
     * Gets the base url of the Switchboard API.
     *
     * @returns The url of the API.
     */
    static std::string getAPIUrl();

    /**
     * Gets the app ID value that was used to initialize the SDK.
     *
     * @returns The app ID value.
     */
    WASM static std::string getAppID();

    /**
     * Gets the app secret value that was used to initialize the SDK.
     *
     * @returns The app secret value.
     */
    WASM static std::string getAppSecret();

    /**
     * Gets the path to the temporary directory.
     *
     * @returns The path to the temporary directory.
     */
    static std::string getTemporaryDirectoryPath();

    /**
     * Sets the path to the temporary directory.
     *
     * @param path The path to the temporary directory.
     */
    static void setTemporaryDirectoryPath(const std::string& path);

    /**
     * Indicates the maximum number of frames that can be accommodated in the audio buffer
     * It's typically used to determine the buffer size needed for audio processing tasks.
     *
     * @return uint The maximum number of frames that can be accommodated in the audio buffer.
     */
    static uint getMaxNumberOfFrames();

    /**
     * Gets the shared license manager instance.
     *
     * @returns The shared license manager instance.
     */
    static LicenseManager* getLicenseManager();

private:
    static bool initialized;
    static std::string appID;
    static std::string appSecret;
    static std::string bundleID;
    static std::string deviceID;
    static std::string apiURL;
    static std::string temporaryDirectoryPath;
    static LicenseManager* licenseManager;
};

}
