//
//  Config.hpp
//  SwitchboardSDK
//
//  Created by Balazs Kiss on 2025. 01. 21..
//

#pragma once

#include <any>
#include <map>
#include <string>
#include <vector>
#include <optional>

namespace switchboard {

/**
 * Configuration class to store and retrieve configuration values.
 */
class Config {
public:
    /**
     * Constructor.
     * @param config The configuration map.
     */
    Config(const std::map<std::string, std::any>& config);

    /**
     * Constructor.
     * @param jsonString The JSON string.
     */
    Config(const std::string& jsonString);

    /**
     * Gets the configuration map.
     *
     * @return The configuration map.
     */
    [[nodiscard]]
    std::map<std::string, std::any> getValue() const;

    /**
     * Gets a string value from the configuration map.
     *
     * @param key The key.
     *
     * @return The value.
     */
    [[nodiscard]] std::string getString(const std::string& key) const;

    /**
     * Gets an integer value from the configuration map.
     *
     * @param key The key.
     *
     * @return The value.
     */
    [[nodiscard]] int getInt(const std::string& key) const;

    /**
     * Gets an unsigned integer value from the configuration map.
     *
     * @param key The key.
     *
     * @return The value.
     */
    [[nodiscard]] unsigned int getUInt(const std::string& key) const;

    /**
     * Gets a double value from the configuration map.
     *
     * @param key The key.
     *
     * @return The value.
     */
    [[nodiscard]] double getDouble(const std::string& key) const;

    /**
     * Gets a value from the configuration map.
     *
     * @tparam T The type of the value.
     * @param key The key.
     *
     * @return The value.
     */
    template <typename T>
    T get(const std::string& key) const;

    /**
     * Converts an any value to a string. Supports both std::string and const char*.
     *
     * @param value The value.
     *
     * @return The string representation of the value.
     */
    static std::string toString(const std::any& value);

    /**
     * Converts an any value to an integer.
     *
     * @param value The value.
     *
     * @return The integer representation of the value.
     */
    static int toInt(const std::any& value);

    /**
     * Converts an any value to an unsigned integer.
     *
     * @param value The value.
     *
     * @return The unsigned integer representation of the value.
     */
    static unsigned int toUInt(const std::any& value);

    /**
     * Converts an any value to a float.
     *
     * @param value The value.
     *
     * @return The float representation of the value.
     */
    static float toFloat(const std::any& value);

    /**
     * Converts an any value to a double.
     *
     * @param value The value.
     *
     * @return The double representation of the value.
     */
    static double toDouble(const std::any& value);

    /**
     * Converts an any value to a boolean.
     *
     * @param value The value.
     *
     * @return The boolean representation of the value.
     */
    static bool toBool(const std::any& value);

    /**
     * Converts an any value to a vector.
     *
     * @tparam T The type of the vector.
     * @param value The value.
     *
     * @return The vector representation of the value.
     */
    template <typename T>
    static std::vector<T> toVector(const std::any& value);

    /**
     * @brief Finds a string in the configuration map.
     * @details If the preferred key is found, the value is returned. Otherwise, the first string value is returned.
     * 
     * @param config The configuration map.
     * @param preferredKey The preferred key for the string value.
     * 
     * @return The string value if found, otherwise std::nullopt.
     */
    static std::optional<std::string> findString(const std::map<std::string, std::any>& config, const std::string& preferredKey);

    /**
     * Converts a JSON string to a map.
     *
     * @param jsonString The JSON string.
     *
     * @return The map.
     */
    static std::map<std::string, std::any> jsonToMap(const std::string& jsonString);

private:
    std::map<std::string, std::any> config;
};

}
