//
//  Result.hpp
//  SwitchboardSDK
//
//  Created by Balazs Kiss on 2025. 01. 24..
//

#pragma once

#include <optional>
#include <string>
#include <variant>

namespace switchboard {

/**
 * @brief A class that represents the result of an operation that can either succeed or fail.
 */
template <typename T>
class Result {
public:
    struct Error {
        /// @brief Error message.
        std::string message;

        /**
         * @brief Error constructor.
         *
         * @param msg Error message.
         */
        Error(std::string msg) : message(std::move(msg)) {}
    };

private:
    using ValueType = std::conditional_t<std::is_void_v<T>, std::monostate, T>;
    std::variant<ValueType, Error> result;

public:
    explicit Result(ValueType value) : result(std::move(value)) {}
    explicit Result(Error error) : result(std::move(error)) {}

    /**
     * @brief Returns true if the operation was successful.
     * 
     * @return True if the operation was successful.
     */
    [[nodiscard]] bool isSuccess() const {
        return std::holds_alternative<ValueType>(result);
    }

    /**
     * @brief Returns true if the operation failed.
     * 
     * @return True if the operation failed.
     */
    [[nodiscard]] bool isError() const {
        return std::holds_alternative<Error>(result);
    }

    /**
     * @brief Returns the value of the operation if it was successful.
     * 
     * @return The value of the operation if it was successful.
     */
    [[nodiscard]] std::optional<ValueType> value() const {
        if (isSuccess()) {
            return std::get<ValueType>(result);
        }
        return std::nullopt;
    }

    /**
     * @brief Returns the error of the operation if it failed.
     * 
     * @return The error of the operation if it failed.
     */
    [[nodiscard]] std::optional<Error> error() const {
        if (isError()) {
            return std::get<Error>(result);
        }
        return std::nullopt;
    }
};

/**
 * @brief Creates a successful result with a value.
 * 
 * @param value The value of the successful result.
 * 
 * @return A successful result with the given value.
 */
template <typename T>
Result<T> makeSuccess(T value) {
    return Result<T>(std::move(value));
}

/**
 * @brief Creates a successful result without a value.
 * 
 * @return A successful result without a value.
 */
inline Result<void> makeSuccess() {
    return Result<void>(std::monostate {});
}

/**
 * @brief Creates an error result with an error code and message.
 * 
 * @param errorMessage The error message.
 * 
 * @return An error result with the given error code and message.
 */
template <typename T>
Result<T> makeError(std::string errorMessage) {
    return Result<T>(typename Result<T>::Error(std::move(errorMessage)));
}

}
