//
//  AudioBusList.hpp
//  SwitchboardSDK
//
//  Created by Balazs Kiss on 2022. 06. 23..
//  Copyright © 2022. Synervoz Inc. All rights reserved.
//

#pragma once

#include "AudioBus.hpp"
#include "AudioBusFormatList.hpp"

#include <vector>

namespace switchboard {

/**
 * AudioBusList abstract superclass.
 * @brief Holds a list of AudioBus instances.
 * @see AudioBusArray
 * @see AudioBusVector
 */
struct AudioBusList {
public:
    /**
     * @brief Default constructor for AudioBusList.
     */
    AudioBusList() = default;

    /**
     * @brief Default destructor for AudioBusList.
     */
    virtual ~AudioBusList() = default;

    /**
     * @brief Gets the formats of the audio buses.
     *
     * @returns An AudioBusFormatList instance that describes the formats of the audio buses.
     */
    virtual AudioBusFormatList& getFormat() = 0;

    /**
     * @brief Gets the number of audio buses.
     *
     * @returns The number of audio buses.
     */
    virtual uint getNumberOfBuses() const = 0;

    /**
     * @brief Gets an audio bus by its index.
     *
     * @param busIndex The index of the audio bus.
     *
     * @returns The AudioBus instance.
     */
    virtual AudioBus& getBus(const uint busIndex) const = 0;

    /**
     * @brief Copies another AudioBusList instance.
     * @details The number of buses has to match in both instances.
     *
     * @param srcAudioBusList The other AudioBusList instance where data is copied from.
     */
    void copyFrom(const AudioBusList& srcAudioBusList) const;

    /**
     * @brief Sets the content of all audio buses to zero.
     */
    void clear() const;

    AudioBusList(const AudioBusList&) = delete;
    AudioBusList& operator=(const AudioBusList&) = delete;
};

}
