//
//  AudioBusArray.hpp
//  SwitchboardSDK
//
//  Created by Balazs Kiss on 03/03/2024.
//

#pragma once

#include "AudioBusFormatArray.hpp"

#include <array>
#include <switchboard_core/AudioBusList.hpp>

namespace switchboard {

/**
 * AudioBusArray class.
 * @brief Holds a list of AudioBus instances and allocates memory on the stack.
 */
template <std::size_t N>
class AudioBusArray : public AudioBusList {
public:
    /**
     * @brief Creates an AudioBusArray instance.
     * @details Allocates enough memory on the stack to hold the AudioBus instances.
     */
    AudioBusArray() {};

    /**
     * @brief Creates an AudioBusArray instance with a single audio bus.
     * @details Allocates memory on the stack for one AudioBus instance.
     *
     * @param audioBus The first audio bus in the AudioBusArray.
     */
    AudioBusArray(AudioBus& audioBus);

    /**
     * @brief Creates an AudioBusArray instance with a single audio bus from an audio buffer.
     * @details Allocates memory on the stack.
     *
     * @param buffer The audio buffer for the first audio bus in the AudioBusArray.
     */
    AudioBusArray(AudioBuffer<float>& buffer);

    // MARK: Overridden methods

    uint getNumberOfBuses() const override;
    AudioBus& getBus(const uint busIndex) const override;
    AudioBusFormatList& getFormat() override;

private:
    std::array<AudioBus, N> buses;
    AudioBusFormatArray<N> format;
};

using EmptyAudioBusArray = AudioBusArray<0>;

}
