//
//  AudioGraphNodeState.hpp
//  SwitchboardSDK
//
//  Created by Balázs Kiss on 2022. 04. 13..
//  Copyright © 2022. Synervoz Inc. All rights reserved.
//

#pragma once

#include "AudioGraphNodeConnection.hpp"

#include <switchboard_core/AudioBusFormatList.hpp>
#include <switchboard_core/AudioBusList.hpp>
#include <switchboard_core/Switchboard.hpp>

namespace switchboard {

/**
 * AudioGraphNodeState class.
 * @brief Holds the state of an audio node inside the AudioGraph instance.
 */
struct AudioGraphNodeState {
public:
    /// @brief Input format of the audio node's buses.
    AudioBusFormatList* inputFormat;

    /// @brief Output format of the audio node's buses.
    AudioBusFormatList* outputFormat;

    /// @brief Input buses of the audio node.
    AudioBusList* inAudioBuses;

    /// @brief Output buses of the audio node.
    AudioBusList* outAudioBuses;

    /// @brief Connections connected to the node's input.
    std::vector<AudioGraphNodeConnection*> inputConnections;

    /// @brief Connections connected to the node's output.
    std::vector<AudioGraphNodeConnection*> outputConnections;

    /**
     * @brief Creates an AudioGraphNodeState instance.
     *
     * @param numberOfInputBuses The number of input buses.
     * @param numberOfOutputBuses The number of output buses.
     */
    AudioGraphNodeState(const uint numberOfInputBuses, const uint numberOfOutputBuses);

    /**
     * @brief AudioGraphNodeState destructor.
     */
    ~AudioGraphNodeState();

    /**
     * @brief Returns the number of input buses for the node.
     *
     * @returns The number of input buses.
     */
    uint getNumberOfInputBuses() const;

    /**
     * @brief Returns the number of output buses for the node.
     *
     * @returns The number of output buses.
     */
    uint getNumberOfOutputBuses() const;

    /**
     * @brief Returns whether input format is completely set for all buses of the audio node.
     *
     * @return True if input format is fully defined, false otherwise.
     */
    bool getIsInputFormatSet() const;

    /**
     * @brief Returns whether input format is set for any  buses of the audio node.
     *
     * @return True if input format is partly defined, false otherwise.
     */
    bool getIsInputFormatPartlySet() const;

    /**
     * @brief Returns whether output format is completely set for all buses of the audio node.
     *
     * @return True if output format is fully defined, false otherwise.
     */
    bool getIsOutputFormatSet() const;

    /**
     * @brief Returns whether output format is set for any  buses of the audio node.
     *
     * @return True if output format is partly defined, false otherwise.
     */
    bool getIsOutputFormatPartlySet() const;

    AudioGraphNodeState(const AudioGraphNodeState&) = delete;
    AudioGraphNodeState& operator=(const AudioGraphNodeState&) = delete;

private:
    uint numberOfInputBuses;
    uint numberOfOutputBuses;
};

}
