//
//  AudioPlayer.hpp
//  SwitchboardSDK
//
//  Created by Nevo Segal on 28/11/2016.
//  Copyright © 2022 Synervoz Inc. All rights reserved.
//

#pragma once

#include "AudioData.hpp"
#include "AudioPlayerSource.hpp"
#include "Resampler.hpp"
#include "RingBuffer.hpp"

#include <atomic>
#include <cmath>
#include <fstream>
#include <memory>
#include <optional>
#include <switchboard_core/Switchboard.hpp>
#include <switchboard_v2/Codec.hpp>
#include <vector>

namespace switchboard {

/**
 * AudioPlayer class.
 * @brief Plays the loaded audio file.
 */
class AudioPlayer {

public:
    /**
     * @brief AudioPlayer constructor.
     */
    AudioPlayer();

    /**
     * @brief AudioPlayer destructor.
     */
    virtual ~AudioPlayer() = default;

    /**
     *  @brief Loads an entire audio file at the given path. Stops playback.
     *
     *  @param path The path of the encoded audio file to load.
     *  @param format The audio file format.
     *
     *  @returns True if loading of the audio file was successful, false otherwise.
     */
    bool load(const std::string& path, Codec format);

    /**
     *  @brief Loads encoded audio data. Stops playback.
     *
     *  @param inputData Vector containing the encoded audio data.
     *  @param format The audio format.
     *
     *  @returns True if loading was successful, false otherwise.
     */
    bool load(const std::vector<uint8>& inputData, Codec format);

    /**
     * @brief Loads audio data from an AudioBuffer instance.
     *
     * @param audioBuffer The audio buffer containing the audio data.
     * @param copyData When set to true, the audio data from the audio buffer will be copied and stored in the AudioPlayer instance.
     *
     * @returns True if loading was successful, false otherwise.
     */
    bool load(AudioBuffer<float>& audioBuffer, bool copyData);

    /**
     *  @brief Streams the encoded audio file from the given path. Stops playback.
     *
     *  @param path The path to the encoded audio file to stream.
     *  @param format The audio file format.
     *
     *  @returns True if loading of the audio file was successful, false otherwise.
     */
    bool stream(const std::string& path, Codec format);

    /**
     * @brief Sets the source of the audio player.
     * 
     * @param audioPlayerSource  The source of the audio player.
     */
    void setSource(AudioPlayerSource& audioPlayerSource);

    /**
     *  @brief Starts the audio playback.
     */
    void play();

    /**
     *  @brief Pauses audio playback.
     */
    void pause();

    /**
     *  @brief Stops audio playback.
     */
    void stop();

    /**
     *  @brief Indicates whether the audio player is currently playing.
     *
     *  @returns True when it is playing, false when it is not.
     */
    bool isPlaying() const;

    /**
     *  @brief Indicates whether looping of the audio file is enabled.
     *
     *  @returns True when it is enabled, false when it is not.
     */
    bool isLoopingEnabled() const;

    /**
     *  @brief Enables or disables looping of the loaded audio file.
     *
     *  @param enabled True when it is enabled, false when it is not.
     */
    void setLoopingEnabled(bool enabled);

    /**
     * The duration of the currently playing media.
     *
     * @returns The duration of the media in seconds.
     */
    double getDuration() const;

    /**
     *  Indicates the current position of the playback head.
     *
     *  @returns Returns the elapsed time in seconds.
     */
    double getPosition() const;

    /**
     * Sets the playback head to the desired position.
     *
     * @param position The desired position in seconds.
     */
    void setPosition(double position);

    /**
     *  Indicates the current start position of the playback.
     *
     *  @returns Returns the current start position in seconds.
     */
    double getStartPosition() const;

    /**
     * Sets the starting position of the playback.
     * Only takes effect after audio is loaded.
     *
     * @param startPosition The desired position in seconds.
     */
    void setStartPosition(double startPosition);

    /**
     *  Indicates the current end position of the playback.
     *
     *  @returns Returns the current end position in seconds.
     */
    double getEndPosition() const;

    /**
     * Sets the ending position of the playback.
     * Only takes effect after audio is loaded.
     *
     * @param endPosition The desired position in seconds.
     */
    void setEndPosition(double endPosition);

    /**
     * @brief Gets the sample rate of the current audio source. SAMPLE_RATE_UNDEFINED is returned if there is none.
     *
     * @returns The current audio source's sample rate.
     */
    uint getSourceSampleRate() const;

    /**
     * @brief Gets the number of channels of the current audio source. NUM_CHANNELS_UNDEFINED is returned if there is none.
     *
     * @returns The current audio source's number of channels.
     */
    uint getSourceNumberOfChannels() const;

    /**
     *  @brief Retrieves the number of buffered frames currently in the AudioPlayer.
     *
     *  The count returned by this method is indicative of the internal state of the AudioPlayer's
     *  buffering system and can be used for diagnostics or managing the flow of data through the
     *  AudioPlayer.
     *
     *  @return The number of buffered frames in the AudioPlayer.
     */
    int getNumberOfBufferedFrames() const;

    /**
     *  @brief Renders the loaded audio data into the given audio buffer.
     *
     *  @param outputBuffer The buffer that will contain the output samples.
     *  @param numberOfFrames The number of frames to put in the output buffer.
     *  @param numberOfChannels The number of channels to put in the output channels.
     *  @param sampleRate The sample rate in which to play the recording.
     */
    virtual void
        process(float* outputBuffer, const uint numberOfFrames, const uint numberOfChannels, const uint sampleRate);

protected:
    std::atomic<bool> playing;
    bool loopingEnabled;

    std::unique_ptr<switchboard::Resampler> resampler;
    std::unique_ptr<RingBuffer<float>> ringBuffer;
    AudioData<float> audioChunk;
    AudioData<float> resampledAudioChunk;
    uint currentSampleRate;

    std::unique_ptr<AudioPlayerSource> internalSource;
    AudioPlayerSource* currentSource;
};

}
