//
//  AudioPlayerNode.hpp
//  SwitchboardSDK
//
//  Created by Nádor Iván on 2022. 08. 30..
//

#pragma once

#include "AudioData.hpp"
#include "AudioPlayer.hpp"
#include "StringCallbackParameter.hpp"

#include <memory>
#include <switchboard_core/SingleBusAudioSourceNode.hpp>

namespace switchboard {

/**
 * AudioPlayerNode class.
 * @brief AudioNode that plays the loaded audio file.
 */
class AudioPlayerNode : public SingleBusAudioSourceNode {
public:
    SB_WASM_EXPORT(AudioPlayerNode)

    /**
     * @brief Creates a AudioPlayerNode instance.
     */
    SB_WASM AudioPlayerNode();

    /**
     * @brief Creates a AudioPlayerNode instance with parameters config.
     */
    AudioPlayerNode(const std::map<std::string, std::any>& config);

    /**
     * @brief AudioPlayerNode destructor
     */
    SB_WASM ~AudioPlayerNode();

    /**
     *  @brief Loads an entire audio file from the given path to the memory. Stops playback.
     *
     *  @param path The path to the audio file to load.
     *  @param format The audio recording format.
     *
     *  @returns True when loading of the audio file was successful, false otherwise.
     */
    bool load(const std::string& path, Codec format);

    /**
     * @brief Loads encoded audio data. Stops playback.
     *
     * @param inputData Vector containing the encoded audio data.
     * @param format The audio encoding format.
     *
     * @returns True when loading the audio data was successful, false otherwise.
     */
    bool load(const std::vector<uint8>& inputData, Codec format);

    /**
     * @brief Loads audio data from an AudioBuffer instance.
     *
     * @param audioBuffer The audio buffer containing the audio data.
     * @param copyData When set to true, the audio data from the audio buffer will be copied and stored in the AudioPlayerNode instance.
     *
     * @returns True when loading the audio data was successful, false otherwise.
     */
    SB_WASM bool load(AudioBuffer<float>& audioBuffer, bool copyData);

    /**
     *  @brief Starts streaming an audio file from the given path. Stops playback.
     *  @details Using this method, instead of load, is more memory efficient as it does not load the entire audio file to memory. However, seeking the audio might take longer.
     *
     *  @param path The path to the audio file to stream.
     *  @param format The audio recording format.
     *
     *  @returns True when streaming of the audio file could be started, false otherwise.
     */
    bool stream(const std::string& path, Codec format);

    /**
     *  @brief Sets the audio source to be played.
     *
     *  @param source The audio source to be played.
     */
    void setSource(AudioPlayerSource& source);

    /**
     *  @brief Starts the audio playback.
     */
    SB_WASM void play();

    /**
     *  @brief Pauses the audio playback.
     */
    SB_WASM void pause();

    /**
     *  @brief Stops the audio playback.
     */
    SB_WASM void stop();

    /**
     *  @brief Indicates whether the system is currently playing.
     *
     *  @returns True when it is playing, false when it is not.
     */
    SB_WASM bool isPlaying() const;

    /**
     *  @brief Indicates whether looping of the audio file is enabled.
     *
     *  @returns True when it is enabled, false when it is not.
     */
    SB_WASM bool isLoopingEnabled() const;

    /**
     * @brief Gets the number of channels that the audio node will output. Default: 2 (stereo)
     *
     * @returns The output number of channels.
     */
    SB_WASM uint getNumberOfChannels() const;

    /**
     * @brief Sets the number of channels that the audio node will output.
     *
     * @param numberOfChannels The new number of channels setting.
     */
    SB_WASM void setNumberOfChannels(const uint numberOfChannels);

    /**
     * @brief Gets the sample rate of the current audio source. SAMPLE_RATE_UNDEFINED is returned if there is none.
     *
     * @returns The current audio source's sample rate.
     */
    SB_WASM uint getSourceSampleRate() const;

    /**
     * @brief Gets the number of channels of the current audio source. NUM_CHANNELS_UNDEFINED is returned if there is none.
     *
     * @returns The current audio source's number of channels.
     */
    SB_WASM uint getSourceNumberOfChannels() const;

    /**
     * @brief Indicates whether looping of the audio file is enabled.
     *
     * @param loopingEnabled True when it is enabled, false when it is not.
     */
    SB_WASM void setLoopingEnabled(bool loopingEnabled);

    /**
     *  @brief Indicates the current position of the playback head.
     *
     *  @returns Returns the elapsed time in seconds.
     */
    SB_WASM double getPosition() const;

    /**
     * @brief The duration of the currently playing media.
     *
     * @returns The duration of the media in seconds.
     */
    SB_WASM double getDuration() const;

    /**
     * @brief Sets the playback head to the desired position.
     *
     * @param position The desired position in seconds.
     */
    SB_WASM void setPosition(double position);

    /**
     *  @brief Indicates the current start position of the playback.
     *
     *  @returns Returns the current start position in seconds.
     */
    double getStartPosition() const;

    /**
     * @brief Sets the starting position of the playback.
     * Only takes effect after audio is loaded.
     *
     * @param startPosition The desired position in seconds.
     */
    void setStartPosition(double startPosition);

    /**
     *  @brief Indicates the current end position of the playback.
     *
     *  @returns Returns the current end position in seconds.
     */
    double getEndPosition() const;

    /**
     * @brief Sets the ending position of the playback.
     * Only takes effect after audio is loaded.
     *
     * @param endPosition The desired position in seconds.
     */
    void setEndPosition(double endPosition);

    /**
     * @brief Gets the audio file path string parsed via constructor that takes parameter config.
     *
     * @returns The file path of the audio file.
     */
    std::string getAudioFilePath() const;

    // MARK: Overridden methods

    bool setBusFormat(AudioBusFormat& busFormat) override;
    bool produce(AudioBus& bus) override;
    Result<std::any> callAction(const std::string& actionName, const std::map<std::string, std::any>& params) override;

private:
    void createParameters();
    void setAudioFilePath(const std::string modelFilePath);
    std::unique_ptr<AudioPlayer> audioPlayer;
    uint numberOfChannels;
    AudioData<float> temporaryInterleavedStereoBuffer;
    std::string audioFilePath;
};

}
