//
//  BoolCallbackParameter.hpp
//  SwitchboardSDK
//
//  Created by Balazs Kiss on 2023. 05. 26..
//

#pragma once

#include "BoolParameter.hpp"

namespace switchboard {

/**
 * @brief BoolCallbackParameter class. A boolean parameter class that wraps a getter and a setter methods.
 */
template <typename T>
class BoolCallbackParameter : public BoolParameter {
public:
    typedef bool (T::*BoolGetter)() const;
    typedef void (T::*BoolSetter)(bool);

    /**
     * @brief BoolCallbackParameter constructor.
     *
     * @param id The unique identifier of the parameter.
     * @param name The name of the parameter.
     * @param description The description of the parameter.
     * @param instance Pointer to the instance that the getter and setter methods belong to.
     * @param getter The getter method for the parameter.
     * @param setter The setter method for the parameter.
     */
    BoolCallbackParameter(
        const std::string& id,
        const std::string& name,
        const std::string& description,
        T* instance,
        BoolGetter getter,
        BoolSetter setter
    ) :
        BoolParameter(id, name, description) {
        this->instance = instance;
        this->getter = getter;
        this->setter = setter;
    }

    // MARK: Overridden methods

    bool getValue() const override {
        return (instance->*getter)();
    }

    void setValue(const bool newValue) override {
        (instance->*setter)(newValue);
    }

private:
    T* instance;
    BoolGetter getter;
    BoolSetter setter;
};

}
