//
//  BufferRegulator.hpp
//  SwitchboardSDK
//
//  Created by Balázs Kiss on 2022. 02. 16..
//  Copyright © 2022. Synervoz Inc. All rights reserved.
//

#pragma once

#include "AudioData.hpp"
#include "RingBuffer.hpp"

#include <memory>
#include <switchboard_core/Switchboard.hpp>

namespace switchboard {

using AudioCallbackFunction = bool (*)(
    float** buffers,
    const uint inputChannels,
    const uint outputChannels,
    const uint numberOfFrames,
    const uint sampleRate,
    void* userPointer
);

/**
 * BufferRegulator class.
 * @brief This class can be used to turn variable sized audio buffer stream into a stabilized constant length audio buffer stream.
 * @details The BufferRegulator's process function can receive any number of frames, but the callback function is always called with outputBufferMs long buffers.
 */
class BufferRegulator {
public:
    /**
     * @brief Creates a BufferRegulator instance.
     *
     * @param outputBufferMs Buffer duration that is received by the callback function.
     * @param audioCallback The callback function that receives the duration regulated audio buffers.
     */
    BufferRegulator(const uint outputBufferMs, const AudioCallbackFunction audioCallback);

    /**
     * @brief Resets the buffer regulator's internal state.
     */
    void reset();

    /**
     * @brief The audio process function.
     * @details When sample rate changes, the BufferRegulator resets its state and clears its internal buffers.
     *
     * @param buffers [inout] The audio buffer.
     * @param inputChannels Number of input channels.
     * @param outputChannels Number of output channels.
     * @param sampleRate Sample rate of audio data.
     * @param userPointer Custom pointer that is passed to the callback function.
     *
     * @returns true if audio data has been written to buffer, false otherwise.
     */
    bool process(
        float** buffers,
        const uint inputChannels,
        const uint outputChannels,
        const uint numberOfFrames,
        const uint sampleRate,
        void* userPointer
    );

private:
    uint outputBufferMs;
    AudioCallbackFunction audioCallback;

    std::unique_ptr<RingBuffer<float>> captureBuffer;
    std::unique_ptr<RingBuffer<float>> renderBuffer;
    uint previousSampleRate;
    uint previousNumberOfInputChannels;
    uint previousNumberOfOutputChannels;

    AudioData<float> tempBufferChannel0;
    AudioData<float> tempBufferChannel1;
    AudioData<float> tempMonoBuffer;
    AudioData<float> tempInterleavedBuffer;
};

}
