//
//  FileWriter.hpp
//  SwitchboardSDK
//
//  Created by Nádor Iván on 2022. 03. 25..
//  Copyright © 2022. Synervoz Inc. All rights reserved.
//

#pragma once

#include <fstream>
#include <switchboard_core/Switchboard.hpp>
#include <vector>

namespace switchboard {

/**
 * FileWriter class for writing raw data to a file.
 * @brief Writes a file to the file system.
 * @details Resource acquisition happens when constructing FileWriter. Destruction ensures that the file is closed.
 * @details Reading from the file can happen after closing it (destructing FileWriter, by scoping it, calling reset on a unique_ptr, etc.).
 */
class FileWriter {
public:
    /**
     *  @brief Writes contents of input buffer to a file at filePath.
     *
     *  @param filePath Path of the file to write to.
     *  @param buffer Buffer with the input data.
     *  @param size Size of the input buffer in bytes.
     *
     *  @return True if the data was written successfully, false otherwise.
     */
    static const bool write(const std::string filePath, const int8* buffer, const uint64 size);

    /**
     *  @brief Tries to create FileWriter by opening an output stream. Throws std::runtime_error if could not open file.
     *
     *  @param filePath Destination file path.
     *  @param allowOverwrite Indicates if only non-existent file should be opened. If this is false and there is already a file at filePath, throw std::runtime_error.
     */
    FileWriter(const std::string& filePath, bool allowOverwrite = true);

    /**
     *  @brief FileWriter destructor.
     */
    ~FileWriter();

    FileWriter(const FileWriter&) = delete;

    /**
     *  @brief Writes content of buffer in the destination file.
     *
     *  @param buffer Raw data buffer.
     *  @param size Size of the raw data buffer.
     *  @returns True if write is successful, false otherwise.
     */
    bool write(const int8* buffer, const uint64 size) noexcept;

    /**
     *  @brief Writes content of buffer in the destination file.
     *
     *  @param buffer Raw data vector.
     *  @returns True if write is successful, false otherwise.
     */
    bool write(const std::vector<int8>& buffer) noexcept;

private:
    std::ofstream outputStream;
};

}
