//
//  Graph.hpp
//  SwitchboardSDK
//
//  Created by Balázs Kiss on 2022. 03. 01..
//  Copyright © 2022. Synervoz Inc. All rights reserved.
//

#pragma once

#include <map>
#include <set>
#include <switchboard_core/Switchboard.hpp>
#include <vector>

namespace switchboard {

/**
 * Graph classes.
 * @brief Represents a general purpose DAG (Directed Acyclic Graph) with nodes and edges.
 */
class Graph {
public:
    /**
     * @brief Creates a Graph instance.
     */
    Graph();

    /**
     * @brief Graph destructor.
     */
    ~Graph();

    /**
     * @brief Returns the number of nodes in the graph.
     *
     * @returns The number of nodes.
     */
    uint getNumberOfNodes() const;

    /**
     * @brief Adds a node the graph.
     *
     * @returns The ID of the new node.
     */
    uint addNode();

    /**
     * @brief Removes a node from the graph.
     *
     * @param node The ID of the node to be removed.
     *
     * @returns True if the removal was successful, false otherwise.
     */
    bool removeNode(uint node);

    /**
     * @brief Adds an edge between two nodes.
     *
     * @param startNode The start node.
     * @param endNode The end node.
     *
     * @returns True if edge was added successfully, false otherwise.
     */
    bool addEdge(const uint startNode, const uint endNode);

    /**
     * @brief Removes an edge from between two nodes.
     *
     * @param startNode The start node.
     * @param endNode The end node.
     *
     * @returns True if edge was removed successfully, false otherwise.
     */
    bool removeEdge(const uint startNode, const uint endNode);

    /**
     * @brief Returns out nodes for a given node.
     *
     * @param startNode The node for which the outgoing edges are checked.
     *
     * @returns A vector of the connected nodes.
     */
    const std::vector<uint>& getOutNodes(const uint startNode);

    /**
     * @brief Returns in nodes for a given node.
     *
     * @param endNode The node for which the incoming edges are checked.
     *
     * @returns A vector of the connected nodes.
     */
    const std::vector<uint>& getInNodes(const uint endNode);

    /**
     * @brief Checks whether the graph is cyclic.
     *
     * @returns True if the graph is cyclic, false otherwise.
     */
    bool isCyclic();

private:
    std::vector<std::vector<uint>> adjacencyMatrix;
    std::set<uint> nodes;
    uint numberOfNodesAdded;
    std::vector<uint> nodeListResult;

    bool hasNode(const uint node) const;
    bool isConnected(const uint startNode, const uint endNode) const;

    enum class NodeState { Unprocessed, BeingProcessed, Processed };
    bool checkNodeForCycle(std::map<uint, NodeState>& nodeStates, const uint node);
};

}
