//
//  OfflineGraphRenderer.hpp
//  SwitchboardSDK
//
//  Created by Balazs Kiss on 2022. 12. 15..
//

#pragma once

#include "AudioGraph.hpp"

#include <switchboard_core/SwitchboardObject.hpp>
#include <switchboard_v2/Codec.hpp>

namespace switchboard {

using ProgressUpdateFunction = void (*)(const uint numberOfBuffersProcessed, void* userPointer);

/**
 * OfflineGraphRenderer class.
 * @brief Renders an audio file through an audio graph then saves the output to another audio file.
 */
class OfflineGraphRenderer : public SwitchboardObject {
public:
    /**
     * @brief Audio file for OfflineGraphRenderer.
     */
    struct AudioFile {

        /// File path of the audio file.
        std::string filePath;

        /// Codec of the audio file.
        Codec codec;

        /// Number of channels for the audio bus that belongs to the audio file.
        /// For input files, NUM_CHANNELS_UNDEFINED means that the number of channels of the input audio file will be used.
        /// For output files, numberOfChannels must not be NUM_CHANNELS_UNDEFINED.
        uint numberOfChannels;

        /// Sample rate for the audio bus that belongs to the audio file.
        /// For input files, SAMPLE_RATE_UNDEFINED means that the sample rate of the input audio file will be used.
        /// For output files, SAMPLE_RATE_UNDEFINED means that the audio graph's processing sample rate will be used.
        uint sampleRate;

        /**
         * @brief AudioFile constructor.
         *
         * @param filePath File path of the audio file.
         * @param codec Codec of the audio file.
         * @param numberOfChannels Number of channels for the audio bus that belongs to the audio file.
         * @param sampleRate Sample rate for the audio bus that belongs to the audio file.
         */
        AudioFile(const std::string& filePath, const Codec codec, const uint numberOfChannels, uint sampleRate);

        /**
         * @brief AudioFile constructor.
         *
         * @param config Configuration map.
         */
        AudioFile(const std::map<std::string, std::any>& config);
    };

    /// Input files for the OfflineGraphRenderer. Each input file will appear as an input audio bus on the processed AudioGraph.
    std::vector<AudioFile> inputFiles;

    /// Output files for the OfflineGraphRenderer. Each output file will appear as an output audio bus on the processed AudioGraph.
    std::vector<AudioFile> outputFiles;

    /**
     * OfflineGraphRenderer constructor.
     */
    OfflineGraphRenderer();

    /**
     * OfflineGraphRenderer constructor.
     *
     * @param config Configuration map.
     */
    OfflineGraphRenderer(const std::map<std::string, std::any>& config);

    /**
     * @brief Processes the audio graph with the given input and output files.
     *
     * @param graph The audio graph to process.
     * @param inputFiles The input files to process.
     * @param outputFiles The output files to save the processed audio data.
     *
     * @returns Result<void> indicating success or failure.
     */
    Result<void> process(
        switchboard::AudioGraph& graph,
        const std::vector<AudioFile>& inputFiles,
        const std::vector<AudioFile>& outputFiles
    );

    /**
     * @brief Renders an audio graph.
     *
     * @param graph The audio graph to render.
     *
     * @returns Result<void> indicating success or failure.
     */
    Result<void> processGraph(switchboard::AudioGraph& graph);

    /**
     * @brief Gets the maximum duration of the rendered audio data. The renderer will finish whenever it reads this many seconds from the input audio file or when it reaches the end of the input audio file.
     *
     * @returns The maximum number of seconds to render.
     */
    double getMaxNumberOfSecondsToRender() const;

    /**
     * @brief Sets the maximum number of seconds to render.
     *
     * @param maxNumberOfSecondsToRender The new maximum number of seconds to render.
     */
    void setMaxNumberOfSecondsToRender(const double maxNumberOfSecondsToRender);

    /**
     * @brief Gets the processing sample rate of the audio graph. SAMPLE_RATE_UNDEFINED means that the sample rate of the input audio file will be used. Default: SAMPLE_RATE_UNDEFINED
     *
     * @returns The processing sample rate of the audio graph.
     */
    uint getSampleRate() const;

    /**
     * @brief Sets the processing sample rate of the audio graph.
     *
     * @param sampleRate The new processing sample rate of the audio graph.
     */
    void setSampleRate(const uint sampleRate);

    /**
     * @brief Gets the duration of the audio buffers that will be used to process the audio graph. Default: 10 ms
     *
     * @returns The duration of the audio buffers in milliseconds.
     */
    uint getBufferDurationMs() const;

    /**
     * @brief Sets the duration of the audio buffers that will be used to process the audio graph.
     *
     * @param bufferDurationMs The new duration of the audio buffers in milliseconds.
     */
    void setBufferDurationMs(const uint bufferDurationMs);

    /**
     * @brief Gets the pointer value that will be passed back in the callback functions as a parameter. Default: nullptr
     *
     * @returns The current user pointer.
     */
    void* getUserPointer() const;

    /**
     * @brief Sets the pointer value that will be passed back in the callback functions as a parameter.
     *
     * @param userPointer The new user pointer.
     */
    void setUserPointer(void* userPointer);

    /**
     * @brief Sets the callback function that is called to report back the progress of the rendering process.
     *
     * @param progressUpdateFunction The new progress update function. nullptr can be passed in the clear the currently set function.
     */
    void setProgressUpdateFunction(ProgressUpdateFunction progressUpdateFunction);

    // MARK: Overridden methods
    Result<void> setValue(const std::string& key, const std::any& value) override;
    Result<std::any> getValue(const std::string& key) override;
    Result<std::any> callAction(const std::string& actionName, const std::map<std::string, std::any>& params) override;

private:
    double maxNumberOfSecondsToRender;
    uint sampleRate;
    uint bufferDurationMs;
    void* userPointer;
    ProgressUpdateFunction progressUpdateFunction;
    std::unique_ptr<AudioGraph> audioGraph;
};

}
