//
//  Ramp.hpp
//  SwitchboardSDK
//
//  Created by Balazs Kiss on 2023. 01. 06..
//

#pragma once

#include <switchboard_core/Switchboard.hpp>

namespace switchboard {

/**
 * @brief Linear ramp with fixed duration that can be used for smoothing audio samples.
 * @details The duration of the ramp can be set specified in number of samples or in milliseconds. If milliseconds is used, the sampleRate property needs to be set too.
 */
class Ramp {
public:
    /**
     * @brief Ramp constructor.
     */
    Ramp();

    /**
     * @brief Ramp destructor.
     */
    ~Ramp();

    /**
     * @brief Gets the current value of the ramp.
     *
     * @returns The current value.
     */
    float getValue() const;

    /**
     * @brief Sets the current value of the ramp. Restarts ramping.
     *
     * @param newValue The new value to jump to.
     */
    void setValue(const float newValue);

    /**
     * @brief Gets the current target value.
     *
     * @returns The current target value.
     */
    float getTargetValue() const;

    /**
     * @brief Sets the target value. Restarts ramping.
     *
     * @param newValue The new target value.
     */
    void setTargetValue(const float newValue);

    /**
     * @brief Gets the duration of the ramp in samples.
     *
     * @returns The duration of the ramp in samples.
     */
    uint getRampDurationSamples() const;

    /**
     * @brief Sets the duration of the ramp in samples. Restarts ramping.
     *
     * @param numberOfSamples The new ramp duration in samples.
     */
    void setRampDurationSamples(const uint numberOfSamples);

    /**
     * @brief Gets the duration of the ramp in milliseconds.
     *
     * @returns The duration of the ramp in milliseconds.
     */
    uint getRampDurationMs() const;

    /**
     * @brief Sets the duration of the ramp in milliseconds. Restarts ramping.
     *
     * @param durationMs The new ramp duration in milliseconds.
     */
    void setRampDurationMs(const uint durationMs);

    /**
     * @brief Gets the sample rate.
     *
     * @returns The sample rate.
     */
    uint getSampleRate() const;

    /**
     * @brief Sets the sample rate.
     *
     * @param sampleRate The new sample rate.
     */
    void setSampleRate(const uint sampleRate);

    /**
     * @brief Gets the remaining samples until the target value in the current ramp.
     * @details The value changes after the process call.
     *
     * @returns The remaining samples until the target value in the current ramp.
     */
    uint getRemainingSamplesToTargetValue() const;

    /**
     * @brief Processes the ramp and jumps numberOfSamples steps forward.
     *
     * @param numberOfSamples The number of steps to jump in the ramp.
     *
     * @returns The ramp value before advancing numberOfSamples steps.
     */
    float process(const uint numberOfSamples);

    /**
     * @brief Applies a linear ramp multiplier to a buffer.
     *
     * @param input The input buffer.
     * @param output The output buffer.
     * @param start The start value of the ramp.
     * @param end The end value of the ramp.
     * @param numberOfValues The number of values in the buffers.
     */
    static void rampMultiply(float* input, float* output, float start, float end, uint numberOfValues);

    /**
     * @brief Applies a linear ramp multiplier to an interleaved buffer.
     *
     * @param input The interleaved input buffer.
     * @param output The interleaved output buffer.
     * @param start The start value of the ramp.
     * @param end The end value of the ramp.
     * @param numberOfFrames The number of audio frames in the buffers.
     * @param numberOfChannels The number of audio channels in the buffers.
     */
    static void rampMultiplyInterleaved(
        float* input,
        float* output,
        float start,
        float end,
        uint numberOfFrames,
        uint numberOfChannels
    );

private:
    float value;
    float targetValue;
    uint counter;
    uint durationSamples;
    uint sampleRate;
};

}
