//
//  SwitchboardV3.hpp
//  SwitchboardSDK
//
//  Created by Balazs Kiss on 2025. 01. 22..
//

#pragma once

#include "Config.hpp"
#include "Result.hpp"

#include <any>
#include <functional>
#include <map>
#include <string>

namespace switchboard {

/**
 * @brief A simple interface to the Switchboard SDK.
 */
class SwitchboardV3 {
public:
    using ObjectID = std::string;

    /**
     * @brief Initializes the Switchboard SDK.
     *
     * @param config Configuration map.
     *
     * @return The ID of the initialized Switchboard SDK.
     */
    static Result<SwitchboardV3::ObjectID> initialize(const Config& config);

    /**
     * @brief Deinitializes the Switchboard SDK.
     *
     * @return A successful result if the SDK was deinitialized successfully, an error result otherwise.
     */
    static Result<void> deinitialize();

    /**
     * @brief Creates a new Switchboard engine.
     *
     * @param config Configuration map.
     *
     * @return The ID of the created engine.
     */
    static Result<ObjectID> createEngine(const Config& config);

    /**
     * @brief Creates a new Switchboard graph.
     * 
     * @param config Configuration map.
     * 
     * @return The ID of the created graph.
     */
    static Result<ObjectID> createGraph(const Config& config);

    /**
     * @brief Destroys a Switchboard object.
     * 
     * @param objectID The ID of the object.
     * 
     * @return A successful result if the object was destroyed successfully, an error result otherwise.
     */
    static Result<void> destroyObject(ObjectID objectID);

    /**
     * @brief Calls an action on a Switchboard object.
     *
     * @param objectID The ID of the object.
     * @param actionName The name of the action.
     * @param params The parameters of the action.
     *
     * @return The result of the action.
     */
    static Result<std::any>
        callAction(ObjectID objectID, const std::string& actionName, const std::map<std::string, std::any>& params = {});

    /**
     * @brief Sets a value on a Switchboard object.
     *
     * @param objectID The ID of the object.
     * @param key The key of the value.
     * @param value The value to set.
     *
     * @return A successful result if the value was set successfully, an error result otherwise.
     */
    static Result<void> setValue(ObjectID objectID, const std::string& key, const std::any& value);

    /**
     * @brief Gets a value from a Switchboard object.
     * 
     * @param objectID The ID of the object.
     * @param key The key of the value.
     * 
     * @return The value.
     */
    static Result<std::any> getValue(ObjectID objectID, const std::string& key);

    /**
     * @brief Adds a new event listener to a Switchboard object.
     * 
     * @param objectID The ID of the object.
     * @param eventName The name of the event to listen to.
     * @param callback The callback function to call when the event is fired.
     * 
     * @return The ID of the listener.
     */
    static Result<unsigned int> addEventListener(ObjectID objectID, const std::string& eventName, const std::function<void(const std::any&)>& callback);

    /**
     * @brief Removes an event listener from a Switchboard object.
     * 
     * @param objectID The ID of the object.
     * @param listenerID The ID of the listener to remove.
     * 
     * @return A successful result if the listener was removed successfully, an error result otherwise.
     */
    static Result<void> removeEventListener(ObjectID objectID, unsigned int listenerID);

private:
    class Impl;
    static Impl* pImpl;
};

}
