//
//  Parameter.hpp
//  SwitchboardSDK
//
//  Created by Balazs Kiss on 2023. 05. 25..
//

#pragma once

#include "Switchboard.hpp"

#include <any>
#include <string>

namespace switchboard {

/**
 * @brief Parameter class. Represents an adjustable config value in an audio node.
 */
class Parameter {
public:
    enum class Type { Float, Bool, Int, UInt, String };

    /**
     * @brief Construct a new Parameter object.
     * 
     * @param type The type of the parameter.
     * @param id The unique identifier of the parameter.
     * @param name The name of the parameter.
     * @param description The description of the parameter.
     */
    Parameter(const Type type, const std::string& id, const std::string& name, const std::string& description);

    /**
     * @brief Parameter destructor.
     */
    virtual ~Parameter() = default;

    /**
     * @brief Gets the type of the parameter.
     *
     * @returns The type of the parameter.
     */
    Type getType() const;

    /**
     * @brief Gets the unique identifier of the parameter.
     *
     * @returns The unique identifier of the parameter.
     */
    const std::string& getID() const;

    /**
     * @brief Gets the name of the parameter.
     *
     * @returns The name of the parameter.
     */
    const std::string& getName() const;

    /**
     * @brief Gets the description of the parameter.
     *
     * @returns The description of the parameter.
     */
    const std::string& getDescription() const;

    /**
     * @brief Sets the value of the parameter.
     *
     * @param newValue The new value for the parameter.
     */
    virtual void setValue(std::any newValue) = 0;

    /**
     * @brief Gets the value of the parameter.
     * 
     * @returns The value of the parameter.
     */
    virtual std::any getAnyValue() = 0;

protected:
    Type type;
    std::string id;
    std::string name;
    std::string description;
};

}
