//
//  AudioData.hpp
//  SwitchboardSDK
//
//  Created by Nádor Iván on 2022. 07. 25..
//

#pragma once

#include <switchboard_core/Switchboard.hpp>

namespace switchboard {

/**
 * Audio data class.
 * @brief Provides helper methods to handle raw audio data arrays.
 */
template <typename T>
class AudioData final {
public:
    SB_WASM_EXPORT(AudioData)
    /**
     * @brief Creates an AudioData instance that allocates and owns maxNumberOfChannels * maxNumberOfFrames * sizeof(T) bytes.
     * @details This constructor is not realtime-safe as it allocates memory.
     * @details The allocated buffer is filled with zeros. 
     *
     * @param maxNumberOfChannels The maximum number of channels.
     * @param maxNumberOfFrames The maximum number of frames.
     */
    SB_WASM AudioData(const uint maxNumberOfChannels, const uint maxNumberOfFrames);

    /**
     * @brief AudioData destructor.
     */
    SB_WASM ~AudioData() noexcept;

    /**
     * @brief Returns a pointer to the internal allocated audio memory location.
     *
     * @returns Pointer to the internal allocated audio memory location.
     */
    SB_WASM T* getBuffer();

    /**
     * @brief Returns a pointer to the stacked audio memory location.
     * @details This returns a pointer to a list of per-channel memory locations that have the same memory location as getBuffer().
     * @details This makes it possible to iterate over the channels of the buffer.
     *
     * @returns Pointer to the stacked audio memory location.
     */
    SB_WASM T** getBuffers();

    uint getMaxNumberOfChannels() const {
        return maxNumberOfChannels;
    }

    uint getMaxNumberOfFrames() const {
        return maxNumberOfFrames;
    }

private:
    uint maxNumberOfChannels;
    uint maxNumberOfFrames;

    T* buffer;
    T** buffers;
};

}
