//
//  AudioIO.hpp
//  SwitchboardSDKMacOS
//
//  Created by Iván Nádor on 18/07/2024.
//

#pragma once

#include <map>
#include <optional>
#include <switchboard_core/Switchboard.hpp>
#include <vector>

namespace switchboard {

using AudioProcessingCallback = void (*)(float**, uint, float**, uint, uint, uint, void*);

/**
 * AudioIO class for desktop platforms
 * @brief Provides audio hardware input/output on macOS, Linux and Windows on multiple audio APIs.
 */
class AudioIO {
public:
    /**
     * AudioIO API type
     * @brief API type for the various platform-specific audio APIs.
     */
#if defined(SWITCHBOARD_MAC)
    enum AudioAPI { Core = 1 };
#elif defined(SWITCHBOARD_WINDOWS)
    enum AudioAPI { ASIO = 6, WASAPI = 7 };
#elif defined(SWITCHBOARD_LINUX)
    enum AudioAPI { ALSA = 2, Pulse = 4 };
#endif

    /**
     * AudioIO Error type
     * @brief Error type for the various operations on AudioIO.
     */
    enum Error {
        AudioIONoError = 0, // No error.
        AudioIOWarning, // A non-critical error.
        AudioIOUnknownError, // An unspecified error type.
        AudioIONoDevicesFound, // No devices found on system.
        AudioIOInvalidDevice, // An invalid device ID was specified.
        AudioIODeviceDisconnected, // A device in use was disconnected.
        AudioIOMemoryError, // An error occurred during memory allocation.
        AudioIOInvalidParameter, // An invalid parameter was specified to a function.
        AudioIOInvalidUse, // The function was called incorrectly.
        AudioIODriverError, // A system driver error occurred.
        AudioIOSystemError, // A system error occurred.
        AudioIOThreadError // A thread error occurred.
    };

    /**
     * @brief Get the more verbose string representation of an Error type.
     *
     * @param error The Error type.
     *
     * @returns The string representation of an Error type.
     */
    static std::string getErrorString(Error error);

    /**
     * AudioDevice struct
     * @brief Struct representing a system audio device.
     */
    struct AudioDevice {
        uint id { 0 }; //!< The ID of the audio device. Guaranteed to be unique, 0 if invalid
        std::string name {}; //!< The name of the audio device.
        uint inputChannels {}; //!< The number of available input channels.
        uint outputChannels {}; //!< The number of available output channels.
        bool isDefaultOutput { false }; //!< Shows if this device is the default audio output.
        bool isDefaultInput { false }; //!< Shows if this device is the default audio input.
        std::vector<uint> sampleRates {}; //!< The supported sample rates.
        uint currentSampleRate {}; //!< The current sample rate.
        uint preferredSampleRate {}; //!< The preferred sample rate.

        /**
         * @brief Get the string representation of the AudioDevice.
         *
         * @returns The string representation of the AudioDevice.
         */
        std::string toString() const;

        /**
         * @brief Check if the AudioDevice is an actual valid audio device.
         *
         * @returns True if the AudioDevice is valid.
         */
        bool isValid() const;

        /**
         * @brief Check if the AudioDevice is an input device.
         *
         * @returns True if the AudioDevice is an input device.
         */
        bool isInputDevice() const;

        /**
         * @brief Check if the AudioDevice is an output device.
         *
         * @returns True if the AudioDevice is an output device.
         */
        bool isOutputDevice() const;
    };

    /**
     * StreamParameters struct
     * @brief Struct representing an audio stream parametrization.
     */
    struct StreamParameters {
        /**
         * StreamParameters constructor
         */
        StreamParameters();

        AudioDevice selectedInputDevice {}; //!< The selected input AudioDevice for the stream.
        AudioDevice selectedOutputDevice {}; //!< The selected output AudioDevice for the stream.
        uint numberOfInputChannels { 0 }; //!< The required number of input channels of the stream.
        uint numberOfOutputChannels { 0 }; //!< The required number of output channels of the stream.
        uint preferredBufferSize { 0 }; //!< The preferred buffer size of the stream.
        uint preferredSampleRate { 0 }; //!< The preferred sample rate of the stream.

        /**
         * @brief Get the string representation of the StreamParameters.
         *
         * @returns The string representation of the StreamParameters.
         */
        std::string toString() const;
    };

    /**
     * @brief AudioIO constructor.
     *
     * @param api The requested audio API of the system.
     * @param callback The audio processing callback of the client using the AudioIO class.
     * @param client The client using the AudioIO class. Will be provided in callback.
     */
    AudioIO(AudioAPI api, AudioProcessingCallback callback, void* client);
    ~AudioIO();

    /**
     * @brief Starts the audio input/output.
     *
     * @param enableInput Signifies whether to enable audio input on AudioIO.
     * @param enableOutput Signifies whether to enable audio output on AudioIO.
     * @param parameters The requested stream parameters of the audio stream.
     *
     * @returns AudioIONoError if started successfully or an Error type if not.
     */
    Error start(bool enableInput = true, bool enableOutput = true, StreamParameters parameters = StreamParameters());

    /**
     * @brief Stops the audio input/output.
     */
    void stop();

    /**
     * @brief Get the available audio devices on the system.
     *
     * @returns The available audio devices.
     */
    const std::vector<AudioDevice> getAudioDevices();

    /**
     * @brief Get the current input audio device.
     *
     * @returns The current input audio device.
     */
    const std::optional<AudioDevice> getCurrentInputDevice();

    /**
     * @brief Get the current output audio device.
     *
     * @returns The current output audio device.
     */
    const std::optional<AudioDevice> getCurrentOutputDevice();

private:
    class Internals;
    std::unique_ptr<Internals> internals;

    static int audioCallback(
        void* outputBuffer,
        void* inputBuffer,
        unsigned int numberOfFrames,
        double streamTime,
        uint status,
        void* clientData
    );

    static void errorCallback(unsigned long errorType, const std::string& errorText);

    std::map<uint, AudioDevice> queryAudioDevices();
    uint getPreferredSampleRate(const AudioDevice& inputDevice, const AudioDevice& outputDevice);
    Error setupStreamParameters(StreamParameters parameters);
};

}
