//
//  AudioPlayerSource.hpp
//  SwitchboardSDK
//
//  Created by Iván Nádor on 2023. 02. 07..
//

#pragma once

#include <switchboard_core/Switchboard.hpp>
#include <vector>

namespace switchboard {

/**
 * AudioPlayerSource interface.
 * @brief Provides an interface to provide audio data in chunks.
 * @details It can be used to implement audio streaming.
 */
struct AudioPlayerSource {
    /**
     * @brief Default destructor.
     */
    virtual ~AudioPlayerSource() = default;

    /**
     *  @brief Gets the sample rate of the audio.
     *
     *  @returns The sample rate of the audio.
     */
    virtual const uint getSampleRate() const = 0;

    /**
     *  @brief Gets the number of channels of the audio.
     *
     *  @returns The number of channels of the audio.
     */
    virtual const uint getNumberOfChannels() const = 0;

    /**
     *  @brief Gets the number of frames of the audio.
     *
     *  @returns The number of frames of the audio.
     */
    virtual const uint getNumberOfFrames() const = 0;

    /**
     *  @brief Checks if the provided audio has reached its end.
     *
     *  @returns True if the provided audio has reached its end.
     */
    virtual const bool isFinished() const = 0;

    /**
     *  @brief Indicates whether looping of the audio is enabled.
     *
     *  @returns True when it is enabled, false when it is not.
     */
    virtual const bool isLoopingEnabled() const = 0;

    /**
     *  @brief Enables or disables the looping of the loaded audio.
     *
     *  @param enabled True when it is enabled, false when it is not.
     */
    virtual void setLoopingEnabled(bool enabled) = 0;

    /**
     * @brief Gets the duration of the currently playing media.
     *
     * @returns The duration of the media in seconds.
     */
    virtual const double getDuration() const = 0;

    /**
     *  @brief Indicates the current position of the playhead.
     *
     *  @returns Returns the elapsed time in seconds.
     */
    virtual const double getPosition() const = 0;

    /**
     * @brief Sets the playhead to the desired position.
     *
     * @param position The desired position in seconds.
     */
    virtual void setPosition(double position) = 0;

    /**
     *  Indicates the current start position of the playback.
     *
     *  @returns Returns the current start position in seconds.
     */
    virtual double getStartPosition() const = 0;

    /**
     * Sets the starting position of the playback.
     *
     * @param startPosition The desired position in seconds.
     */
    virtual void setStartPosition(double startPosition) = 0;

    /**
     *  Indicates the current end position of the playback.
     *
     *  @returns Returns the current end position in seconds.
     */
    virtual double getEndPosition() const = 0;

    /**
     * Sets the ending position of the playback.
     *
     * @param endPosition The desired position in seconds.
     */
    virtual void setEndPosition(double endPosition) = 0;

    /**
     *  @brief Gets the next available chunk of audio.
     *
     *  @param outputBuffer The buffer that will contain the output samples.
     *  @param numberOfFrames The number of samples to put in the output buffer.
     *  @returns The number of "useful" audio frames put in outputBuffer.
     */
    virtual const uint read(float* outputBuffer, const uint numberOfFrames) = 0;
};

}
