//
//  DuckingManager.hpp
//  Pods
//
//  Created by Nevo Segal on 13/06/2017.
//
//

#pragma once

#include "DuckingCompressor.hpp"
#include "DuckingManagerDelegate.hpp"

#include <cmath>
#include <map>
#include <switchboard_core/Switchboard.hpp>
#include <vector>

namespace switchboard {

/**
 * DuckingManager class.
 * @brief Calculates and applies ducking based on configuration.
 */
class DuckingManager {
public:
    /**
     * @brief DuckingManager constructor.
     */
    DuckingManager();

    /**
     * @brief DuckingManager destructor.
     */
    ~DuckingManager();

    /**
     * @brief Returns the number of ducking signals.
     *
     * @returns The number of ducking signals.
     */
    uint getDuckingSignalCount() const;

    /**
     * @brief Sets the number of ducking signals.
     *
     * @param numberOfDuckingSignals The new number of ducking signals.
     */
    void setNumberOfDuckingSignals(const uint numberOfDuckingSignals);

    /**
     * @brief Triggers ducking for a ducking signal.
     *
     * @param duckingSignalIndex The index of the ducking signal that is triggering ducking.
     */
    void triggerDucking(const uint duckingSignalIndex = 0);

    /**
     * @brief Processes and applies ducking for the music audio buffer.
     *
     * @param sampleRate the sampleRate of the data to be ducked
     * @param numberOfFrames the number of frames in each input buffer.
     * @param musicAudioBuffer The music audio buffer that will be ducked. Must be a stereo interleaved buffer. Can be nullptr if ducking is only used via the delegate.
     */
    void process(const uint sampleRate, const uint numberOfFrames, float* musicAudioBuffer);

    /**
     * @brief Gets the ducking amount for a ducking signal.
     * Returned value is between 0 and 1.
     *
     * @param duckingSignalIndex The index of the ducking signal.
     *
     * @return The ducking amount for the ducking signal in dBFS.
     */
    float getDuckingAmount(const uint duckingSignalIndex) const;

    /**
     *  @brief Sets the ducking amount for a ducking signal.
     *  The provided value has to be between 0 and 1.
     *
     *  @param newDuckingAmount The new amount.
     *  @param duckingSignalIndex The index of the ducking signal.
     */
    void setDuckingAmount(const float newDuckingAmount, const uint duckingSignalIndex);

    /**
     * @brief Returns the ducking release amount.
     * When the ducking threshold is not exceeded, the ducking amount value will be increased by this value.
     * Default value: 0.5 dB
     *
     * @returns the ducking release amount.
     */
    float getDuckReleaseAmount() const;

    /**
     * @brief Sets the ducking release amount.
     * When the ducking threshold is not exceeded, the ducking amount value will be increased by this value.
     * Default value: 0.5 dB
     *
     * @param newDuckReleaseAmount The new value.
     */
    void setDuckReleaseAmount(const float newDuckReleaseAmount);

    float getDuckReleaseSeconds() const;

    void setDuckReleaseSeconds(const float duckReleaseSeconds);

    /**
     * @brief Returns the number of seconds to hold ducking value.
     * When ducking stops being triggered, ducking will still be held for this many seconds.
     * Default value: 2.0 secs
     *
     * @returns The number of seconds to hold ducking.
     */
    float getNumSecondsToHoldDucking() const;

    /**
     * @brief Sets the number of seconds to hold ducking value.
     *
     * @param numSecondsToHoldDucking The new value.
     */
    void setNumSecondsToHoldDucking(const float numSecondsToHoldDucking);

    /**
     * @brief Gets the current ducking delegate. Can be null.
     *
     * @returns Pointer to the ducking delegate instance.
     */
    DuckingManagerDelegate* getDelegate() const;

    /**
     * @brief Sets the ducking delegate.
     *
     *  @param delegate The new ducking delegate instance. Can be null.
     */
    void setDelegate(DuckingManagerDelegate* delegate);

    /**
     * @brief Gets the compressor.
     *
     * @returns The compressor instance. Can be null.
     */
    DuckingCompressor* getCompressor() const;

    /**
     * @brief Sets the compressor.
     *
     * @param compressor The new compressor instance. Can be null.
     */
    void setCompressor(DuckingCompressor* compressor);

    /**
     * @brief Returns the internal state of the DuckingManager for diagnostics purposes.
     *
     * @returns The internal state.
     */
    std::map<std::string, std::string> getState();

private:
    DuckingManagerDelegate* delegate;
    DuckingCompressor* musicCompressor;

    std::vector<bool> duckingTriggeredForSignal;
    std::map<uint, float> duckingAmountForSignal;
    float currentDuckingAmount;
    float previousDuckingAmount;
    int buffersOverDuckThreshold;
    int64 duckHoldBuffers;
    float numSecondsToHoldDucking;
    int maxDuckedTime;
    int timeDucked;
    float duckReleaseAmount;

    float checkDuckingAmount();
    float getLinearValueFromDbScale(float valueInDb) const;
    float calculateDuckingAmount(uint sampleRate, uint numberOfSamples);
};

}
