//
//  FloatCallbackParameter.hpp
//  SwitchboardSDK
//
//  Created by Balazs Kiss on 2023. 05. 26..
//

#pragma once

#include "FloatParameter.hpp"

namespace switchboard {

/**
 * @brief FloatCallbackParameter class. A float parameter class that wraps a getter and a setter methods.
 */
template <typename T>
class FloatCallbackParameter : public FloatParameter {
public:
    typedef float (T::*FloatGetter)() const;
    typedef void (T::*FloatSetter)(float);

    /**
     * @brief FloatCallbackParameter constructor.
     *
     * @param id The unique identifier of the parameter.
     * @param name The name of the parameter.
     * @param description The description of the parameter.
     * @param instance Pointer to the instance that the getter and setter methods belong to.
     * @param getter The getter method for the parameter.
     * @param setter The setter method for the parameter.
     * @param minimumValue Minimum value for the parameter.
     * @param maximumValue Maximum value for the parameter.
     */
    FloatCallbackParameter(
        const std::string& id,
        const std::string& name,
        const std::string& description,
        T* instance,
        FloatGetter getter,
        FloatSetter setter,
        const float minimumValue,
        const float maximumValue
    ) :
        FloatParameter(id, name, description),
        instance { instance },
        getter { getter },
        setter { setter },
        minimumValue { minimumValue },
        maximumValue { maximumValue } {}

    // MARK: Overridden methods

    float getValue() const override {
        return (instance->*getter)();
    }

    void setValue(const float newValue) override {
        (instance->*setter)(newValue);
    }

    float getMinimumValue() const override {
        return minimumValue;
    }

    float getMaximumValue() const override {
        return maximumValue;
    }

private:
    T* instance;
    FloatGetter getter;
    FloatSetter setter;
    float minimumValue;
    float maximumValue;
};

}
