//
//  IOResampler.hpp
//  SwitchboardAudio
//
//  Created by Balazs Kiss on 2022. 04. 08..
//  Copyright © 2022. Synervoz Inc. All rights reserved.
//

#pragma once

#include "AudioData.hpp"
#include "Resampler.hpp"
#include "RingBuffer.hpp"

#include <memory>

namespace switchboard {

/**
 * @brief Callback function that is called with the desired sample rate.
 */
using IOResamplerCallbackFunction = bool (*)(
    /// Audio buffers [inout].
    float** buffers,
    /// Number of input channels.
    const uint inputChannels,
    /// Number of output channels.
    const uint outputChannels,
    /// Number of frames.
    const uint numberOfFrames,
    /// Sample rate.
    const uint sampleRate,
    /// Custom pointer passed to process function.
    void* userPointer
);

/**
 * IOResampler class.
 * @brief Provides a callback function that is called with resampled audio data.
 * @details Supports mono and stereo input and stereo output.
 * @details The IOResampler contains two resamplers and it works like this:
 * @details [Input audio @ source sample rate] - [Resampler] - [Callback function @ callback sample rate] - [Resampler] - [Output audio @ source sample rate]
 * @details Number of frames is never changed during the process call. Returned audio data might contain zero values initially.
 */
class IOResampler {
public:
    /**
     * @brief Creates an IOResampler instance.
     *
     * @param sourceSampleRate The source sample rate.
     * @param callbackSampleRate The sample rate of the audio data that will be passed to the callback function.
     * @param callback The callback function that processed the resampled audio data.
     */
    IOResampler(const uint sourceSampleRate, const uint callbackSampleRate, const IOResamplerCallbackFunction callback);

    /**
     * @brief Audio process function of IOResampler.
     *
     * @param buffers [inout] The buffers containing the audio data.
     * @param inputIsMono Must be true if the buffers parameter contains mono data, false otherwise.
     * @param numberOfFrames Number of frames in the buffers parameter.
     * @param userPointer Custom pointer value that will be passed to the callback function.
     *
     * @returns True if audio data has been written to buffers, false otherwise.
     */
    bool process(float** buffers, const bool inputIsMono, const uint numberOfFrames, void* userPointer);

private:
    uint sourceSampleRate;
    uint callbackSampleRate;
    float inputSampleRateRatio;
    float outputSampleRateRatio;
    IOResamplerCallbackFunction callback;

    Resampler inputResampler;
    Resampler outputResampler;
    AudioData<float> interleavedBuffer;
    AudioData<float> callbackBufferInterleaved;
    AudioData<float> callbackBufferStacked;
    std::unique_ptr<RingBuffer<float>> outputRingBuffer;
};

}
