//
//  SwitchboardSDK.hpp
//  SwitchboardSDK
//
//  Created by Balazs Kiss on 2023. 01. 06..
//

#pragma once

#include <string>
#include <switchboard_core/Switchboard.hpp>
#include <switchboard_core/SwitchboardObject.hpp>

namespace switchboard {

class LicenseManager;

/**
 * SwitchboardSDK class.
 * @brief Initializes and configures the Switchboard SDK.
 */
class SwitchboardSDK final : public SwitchboardObject {
public:
    SB_WASM_EXPORT(SwitchboardSDK)

    SB_WASM static void initialize(const char* appID, const char* appSecret);

    /**
     * Initializes the Switchboard SDK.
     *
     * @param appID The app ID value that identifies the user of the SDK.
     * @param appSecret The app secret ID.
     */
    static void initialize(const std::string appID, const std::string appSecret);

    /**
     * Gets whether the SDK has been initialized or not.
     *
     * @returns True if the SDK has been initialized, false otherwise.
     */
    static bool isInitialized();

    /**
     * Gets the version number of the SDK.
     *
     * @returns The version number of the SDK.
     */
    static std::string getVersionName();

    /**
     * Gets the build number of the SDK.
     *
     * @returns The build number of the SDK.
     */
    static uint getBuildNumber();

    /**
     * Gets the base url of the Switchboard API.
     *
     * @returns The url of the API.
     */
    static std::string getAPIUrl();

    /**
     * Gets the app ID value that was used to initialize the SDK.
     *
     * @returns The app ID value.
     */
    SB_WASM static std::string getAppID();

    /**
     * Gets the app secret value that was used to initialize the SDK.
     *
     * @returns The app secret value.
     */
    SB_WASM static std::string getAppSecret();

    /**
     * Gets the path to the temporary directory.
     *
     * @returns The path to the temporary directory.
     */
    static std::string getTemporaryDirectoryPath();

    /**
     * Sets the path to the temporary directory.
     *
     * @param path The path to the temporary directory.
     */
    static void setTemporaryDirectoryPath(const std::string& path);

    /**
     * Indicates the maximum number of frames that can be accommodated in the audio buffer
     * It's typically used to determine the buffer size needed for audio processing tasks.
     *
     * @return uint The maximum number of frames that can be accommodated in the audio buffer.
     */
    static uint getMaxNumberOfFrames();

    /**
     * Gets the shared license manager instance.
     *
     * @returns The shared license manager instance.
     */
    static LicenseManager* getLicenseManager();

    static SwitchboardSDK& getInstance();

    // MARK: Overridden methods
    Result<void> setValue(const std::string& key, const std::any& value) override;
    Result<std::any> getValue(const std::string& key) override;

private:
    static SwitchboardSDK* sharedInstance;

    bool initialized;
    std::string appID;
    std::string appSecret;
    std::string bundleID;
    std::string deviceID;
    std::string apiURL;
    std::string temporaryDirectoryPath;
    LicenseManager* licenseManager;

    SwitchboardSDK();
};

}
