//
//  VoiceActivityDetector.hpp
//  SwitchboardSDK
//
//  Created by Gergye Mihaly on 2021. 05. 21..
//  Copyright © 2021. Synervoz Inc. All rights reserved.
//

#pragma once

#include <switchboard_core/AudioBuffer.hpp>
#include <switchboard_core/Switchboard.hpp>

namespace switchboard {

/**
 * @brief Voice Activity Detector
 */
class VoiceActivityDetector {
public:
    enum class Status { VoiceDetected = 0, Hangover = 1, Idle = 2 };

    /**
     * @brief Voice Activity Detector constructor.
     */
    VoiceActivityDetector();

    /**
     * @brief Processes an audio buffer and returns the new VAD status.
     *
     * @param inputBuffer The input audio buffer.
     *
     * @returns The VAD status.
     */
    Status process(AudioBuffer<float>& inputBuffer);

    /**
     * @brief Sets the duration of the VAD hangover in seconds.
     *
     * @param duration The new duration in seconds.
     */
    void setHangoverDuration(float duration);

    /**
     * @brief Returns the current hangover duration in seconds.
     *
     * @returns Hangover duration in seconds.
     */
    float getHangoverDuration() const;

    /**
     * @brief Sets the audio level trigger amount for the VAD.
     *
     * @param threshold The new threshold.
     */
    void setGainTriggerThreshold(float threshold);

    /**
     * @brief Returns the current audio level trigger threshold.
     *
     * @returns Trigger threshold.
     */
    float getGainTriggerThreshold() const;

    /**
     * @brief Sets the amount of seconds the audio signal must be higher than the trigger threshold for the VAD to be activated.
     *
     * @param duration The new duration.
     */
    void setTriggerDuration(float duration);

    /**
     * @brief Returns the current trigger duration.
     *
     * @returns Trigger duration.
     */
    float getTriggerDuration() const;

private:
    // The duration after a VAD trigger the detector emits Hangover status in seconds
    float hangoverDuration = 1.2;
    // Audio signal energy trigger threshold
    float gainTriggerThreshold = 0.2;
    // The number of seconds the signal must be higher than the gainTriggerThreshold
    float triggerDuration = 0;

    // Idle duration after a trigger
    float idleTime = 0;
    // Duration while the VAD is triggered
    float triggeredTime = 0;
};

}
