//
//  VoiceActivityDetectorNode.hpp
//  SwitchboardSDK
//
//  Created by Gergye Mihály on 2023. 06. 05..
//

#pragma once

#include "VoiceActivityDetector.hpp"

#include <functional>
#include <switchboard_core/SingleBusAudioSinkNode.hpp>

namespace switchboard {

using VoiceActivityCallback = std::function<void(VoiceActivityDetector::Status)>;

/** 
 * VoiceActivityDetectorNode class
 * @brief Analyzes the incoming audio and checks whether there was any voice activity.
 */
class VoiceActivityDetectorNode : public SingleBusAudioSinkNode {
public:
    SB_WASM_EXPORT(VoiceActivityDetectorNode)

    /**
     * @brief Creates a VoiceActivityDetectorNode instance.
     */
    SB_WASM VoiceActivityDetectorNode();

    /**
     *  @brief Checks if the node is enabled.
     *
     *  @returns True if the node is enabled.
     */
    bool getIsEnabled() const;

    /**
     *  @brief Enables or disables the node
     *
     *  @param isEnabled True if the node is enabled.
     */
    void setIsEnabled(const bool isEnabled);

    /**
     * @brief Returns the last buffers VAD status.
     *
     * @returns The last VAD status.
     */
    SB_WASM VoiceActivityDetector::Status getStatus();

    /**
     * @brief Sets the duration of the VAD hangover.
     *
     * @param duration The new duration.
     */
    SB_WASM void setHangoverDuration(float duration);

    /**
     * @brief Returns the current hangover duration.
     *
     * @returns Hangover duration.
     */
    SB_WASM float getHangoverDuration() const;

    /**
     * @brief Sets the audio level trigger amount for the VAD.
     *
     * @param threshold The new threshold.
     */
    SB_WASM void setGainTriggerThreshold(float threshold);

    /**
     * @brief Returns the current audio level trigger threshold.
     *
     * @returns Trigger threshold.
     */
    SB_WASM float getGainTriggerThreshold() const;

    /**
     * @brief Sets the amount of seconds the audio signal must be higher than the trigger threshold for the VAD to be activated.
     *
     * @param duration The new duration.
     */
    SB_WASM void setTriggerDuration(float duration);

    /**
     * @brief Returns the current trigger duration.
     *
     * @returns Trigger duration.
     */
    SB_WASM float getTriggerDuration() const;

    void setVoiceActivityCallback(const VoiceActivityCallback& callback);

    // MARK: Overridden methods

    SB_WASM bool setBusFormat(AudioBusFormat& busFormat) override;
    SB_WASM bool consume(AudioBus& bus) override;

private:
    std::unique_ptr<VoiceActivityDetector> detector;
    std::atomic<VoiceActivityDetector::Status> status;
    std::function<void(VoiceActivityDetector::Status)> voiceActivityCallback;
    std::atomic<bool> isEnabled;
    uint sampleCounter;
    uint voiceStartSample;

    void createParameters();
};

} // namespace
